-- Fichero FSQL_F2.sql
----------------------------------------------------------------------
-- * PAQUETES PL/SQL: FSQL_FUNCTIONS y FSQL_FUNCTIONS2
-- * OBJETIVO:
--	# FSQL_FUNTIONS:	Fichero FSQL_F.SQL
--	               	Funciones para MOSTRAR atributos difusos y
--	               	para COMPARAR atributos y constantes difusas
--	               	con los comparadores de Posibilidad y Necesidad:
--	               	FEQ, FGT, FGEQ, FLT, NFLEQ, NFEQ, NFGT, NFGEQ, NFLT y NFLEQ.
--	# FSQL_FUNTIONS2:	Fichero FSQL_F2.SQL
--	               	Funciones para COMPARAR atributos y constantes difusas
--	               	con los comparadores de Posibilidad y Necesidad:
--	               	MGT, MLT, NMGT y NMLT.
-- * AUTOR         : Jos Galindo G.
-- * COMPARADORES DIFUSOS: Se han dividido en dos paquetes debido a la
--   imposibilidad de crear un nico paquete tan grande.

--**************************************************************
--* * * * * * * * *  Paquete  FSQL_FUNCTIONS2  * * * * * * * * *
--**************************************************************
CREATE OR REPLACE PACKAGE FSQL_FUNCTIONS2 AS

--************ Comparador difuso MGT (Much Greater Than) ************--
FUNCTION MGT_crisp_finter (crisp IN NUMBER, X IN NUMBER, Y IN NUMBER,
  margen IN FUZZY_APPROX_MUCH.MARGEN%TYPE,
  much   IN FUZZY_APPROX_MUCH.MUCH%TYPE) RETURN NUMBER;
PRAGMA RESTRICT_REFERENCES (MGT_crisp_finter,WNDS,WNPS,RNPS);

FUNCTION MGT_crisp_aprox (
  crisp IN NUMBER, F1 IN NUMBER, F2 IN NUMBER, F3 IN NUMBER, F4 IN NUMBER,
  margen IN FUZZY_APPROX_MUCH.MARGEN%TYPE,
  much   IN FUZZY_APPROX_MUCH.MUCH%TYPE) RETURN NUMBER;
PRAGMA RESTRICT_REFERENCES (MGT_crisp_aprox,WNDS,WNPS,RNPS);

FUNCTION MGT_crisp_trape (
  crisp  IN NUMBER, F1 IN NUMBER, F2 IN NUMBER, F3 IN NUMBER ,F4 IN NUMBER,
  margen IN FUZZY_APPROX_MUCH.MARGEN%TYPE,
  much   IN FUZZY_APPROX_MUCH.MUCH%TYPE) RETURN NUMBER;
PRAGMA RESTRICT_REFERENCES (MGT_crisp_trape,WNDS,WNPS,RNPS);

FUNCTION MGT_t1_t1 (
  crisp1 IN NUMBER, crisp2 IN NUMBER,
  margen IN FUZZY_APPROX_MUCH.MARGEN%TYPE,
  much   IN FUZZY_APPROX_MUCH.MUCH%TYPE) RETURN NUMBER;
PRAGMA RESTRICT_REFERENCES (MGT_t1_t1,WNDS,WNPS,RNPS);

FUNCTION MGT_t1_t2 (
  crisp  IN NUMBER,
  OBJ IN FUZZY_LABEL_DEF.OBJ#%TYPE, COL IN FUZZY_LABEL_DEF.OBJ#%TYPE,
  F_TYPE IN FUZZY_COL_LIST.F_TYPE%TYPE,
  F1 IN NUMBER, F2 IN NUMBER, F3 IN NUMBER ,F4 IN NUMBER,
  margen IN FUZZY_APPROX_MUCH.MARGEN%TYPE,
  much   IN FUZZY_APPROX_MUCH.MUCH%TYPE) RETURN NUMBER;
PRAGMA RESTRICT_REFERENCES (MGT_t1_t2,WNDS,WNPS,RNPS);

FUNCTION MGT_crisp (
  crisp  IN NUMBER,
  OBJ    IN FUZZY_LABEL_DEF.OBJ#%TYPE, COL IN FUZZY_LABEL_DEF.OBJ#%TYPE,
  F_TYPE IN FUZZY_COL_LIST.F_TYPE%TYPE,
  F1 IN NUMBER, F2 IN NUMBER, F3 IN NUMBER ,F4 IN NUMBER,
  margen IN FUZZY_APPROX_MUCH.MARGEN%TYPE,
  much   IN FUZZY_APPROX_MUCH.MUCH%TYPE) RETURN NUMBER;
PRAGMA RESTRICT_REFERENCES (MGT_crisp,WNDS,WNPS,RNPS);

FUNCTION MGT_inter (
  X IN NUMBER, Y IN NUMBER,
  OBJ    IN FUZZY_LABEL_DEF.OBJ#%TYPE, COL IN FUZZY_LABEL_DEF.OBJ#%TYPE,
  F_TYPE IN FUZZY_COL_LIST.F_TYPE%TYPE,
  F1 IN NUMBER, F2 IN NUMBER, F3 IN NUMBER, F4 IN NUMBER,
  margen IN FUZZY_APPROX_MUCH.MARGEN%TYPE,
  much   IN FUZZY_APPROX_MUCH.MUCH%TYPE) RETURN NUMBER;
PRAGMA RESTRICT_REFERENCES (MGT_inter,WNDS,WNPS,RNPS);

FUNCTION MGT_aprox (
  A1 IN NUMBER, A2 IN NUMBER, A3 IN NUMBER, A4 IN NUMBER,
  OBJ    IN FUZZY_LABEL_DEF.OBJ#%TYPE, COL IN FUZZY_LABEL_DEF.OBJ#%TYPE,
  F_TYPE IN FUZZY_COL_LIST.F_TYPE%TYPE,
  F1 IN NUMBER, F2 IN NUMBER, F3 IN NUMBER, F4 IN NUMBER,
  margen IN FUZZY_APPROX_MUCH.MARGEN%TYPE,
  much   IN FUZZY_APPROX_MUCH.MUCH%TYPE) RETURN NUMBER;
PRAGMA RESTRICT_REFERENCES (MGT_aprox,WNDS,WNPS,RNPS);

FUNCTION MGT_trape (
  T1 IN NUMBER, T2 IN NUMBER, T3 IN NUMBER, T4 IN NUMBER,
  OBJ    IN FUZZY_LABEL_DEF.OBJ#%TYPE, COL IN FUZZY_LABEL_DEF.OBJ#%TYPE,
  F_TYPE IN FUZZY_COL_LIST.F_TYPE%TYPE,
  F1 IN NUMBER, F2 IN NUMBER, F3 IN NUMBER, F4 IN NUMBER,
  margen IN FUZZY_APPROX_MUCH.MARGEN%TYPE,
  much   IN FUZZY_APPROX_MUCH.MUCH%TYPE) RETURN NUMBER;
PRAGMA RESTRICT_REFERENCES (MGT_trape,WNDS,WNPS,RNPS);

FUNCTION MGT (
  OBJ1    IN FUZZY_LABEL_DEF.OBJ#%TYPE, COL1 IN FUZZY_LABEL_DEF.OBJ#%TYPE,
  F_TYPE1 IN FUZZY_COL_LIST.F_TYPE%TYPE,
  F11 IN NUMBER, F21 IN NUMBER, F31 IN NUMBER, F41 IN NUMBER,
  OBJ2    IN FUZZY_LABEL_DEF.OBJ#%TYPE, COL2 IN FUZZY_LABEL_DEF.OBJ#%TYPE,
  F_TYPE2 IN FUZZY_COL_LIST.F_TYPE%TYPE,
  F12 IN NUMBER, F22 IN NUMBER, F32 IN NUMBER ,F42 IN NUMBER,
  margen IN FUZZY_APPROX_MUCH.MARGEN%TYPE,
  much   IN FUZZY_APPROX_MUCH.MUCH%TYPE) RETURN NUMBER;
PRAGMA RESTRICT_REFERENCES (MGT,WNDS,WNPS,RNPS);

--************* Comparador difuso MLT (Much Less Than) *************--
FUNCTION MLT_crisp_finter (crisp  IN NUMBER, X IN NUMBER, Y IN NUMBER,
  margen IN FUZZY_APPROX_MUCH.MARGEN%TYPE,
  much   IN FUZZY_APPROX_MUCH.MUCH%TYPE) RETURN NUMBER;
PRAGMA RESTRICT_REFERENCES (MLT_crisp_finter,WNDS,WNPS,RNPS);

FUNCTION MLT_crisp_aprox (
  crisp  IN NUMBER, F1 IN NUMBER, F2 IN NUMBER, F3 IN NUMBER, F4 IN NUMBER,
  margen IN FUZZY_APPROX_MUCH.MARGEN%TYPE,
  much   IN FUZZY_APPROX_MUCH.MUCH%TYPE) RETURN NUMBER;
PRAGMA RESTRICT_REFERENCES (MLT_crisp_aprox,WNDS,WNPS,RNPS);

FUNCTION MLT_crisp_trape (
  crisp  IN NUMBER, F1 IN NUMBER, F2 IN NUMBER, F3 IN NUMBER ,F4 IN NUMBER,
  margen IN FUZZY_APPROX_MUCH.MARGEN%TYPE,
  much   IN FUZZY_APPROX_MUCH.MUCH%TYPE) RETURN NUMBER;
PRAGMA RESTRICT_REFERENCES (MLT_crisp_trape,WNDS,WNPS,RNPS);

FUNCTION MLT_t1_t1 (
  crisp1 IN NUMBER, crisp2 IN NUMBER,
  margen IN FUZZY_APPROX_MUCH.MARGEN%TYPE,
  much   IN FUZZY_APPROX_MUCH.MUCH%TYPE) RETURN NUMBER;
PRAGMA RESTRICT_REFERENCES (MLT_t1_t1,WNDS,WNPS,RNPS);

FUNCTION MLT_t1_t2 (
  crisp  IN NUMBER,
  OBJ    IN FUZZY_LABEL_DEF.OBJ#%TYPE, COL IN FUZZY_LABEL_DEF.OBJ#%TYPE,
  F_TYPE IN FUZZY_COL_LIST.F_TYPE%TYPE,
  F1 IN NUMBER, F2 IN NUMBER, F3 IN NUMBER ,F4 IN NUMBER,
  margen IN FUZZY_APPROX_MUCH.MARGEN%TYPE,
  much   IN FUZZY_APPROX_MUCH.MUCH%TYPE) RETURN NUMBER;
PRAGMA RESTRICT_REFERENCES (MLT_t1_t2,WNDS,WNPS,RNPS);

FUNCTION MLT_crisp (
  crisp  IN NUMBER,
  OBJ    IN FUZZY_LABEL_DEF.OBJ#%TYPE, COL IN FUZZY_LABEL_DEF.OBJ#%TYPE,
  F_TYPE IN FUZZY_COL_LIST.F_TYPE%TYPE,
  F1 IN NUMBER, F2 IN NUMBER, F3 IN NUMBER ,F4 IN NUMBER,
  margen IN FUZZY_APPROX_MUCH.MARGEN%TYPE,
  much   IN FUZZY_APPROX_MUCH.MUCH%TYPE) RETURN NUMBER;
PRAGMA RESTRICT_REFERENCES (MLT_crisp,WNDS,WNPS,RNPS);

FUNCTION MLT_inter (
  X IN NUMBER, Y IN NUMBER,
  OBJ    IN FUZZY_LABEL_DEF.OBJ#%TYPE, COL IN FUZZY_LABEL_DEF.OBJ#%TYPE,
  F_TYPE IN FUZZY_COL_LIST.F_TYPE%TYPE,
  F1 IN NUMBER, F2 IN NUMBER, F3 IN NUMBER, F4 IN NUMBER,
  margen IN FUZZY_APPROX_MUCH.MARGEN%TYPE,
  much   IN FUZZY_APPROX_MUCH.MUCH%TYPE) RETURN NUMBER;
PRAGMA RESTRICT_REFERENCES (MLT_inter,WNDS,WNPS,RNPS);

FUNCTION MLT_aprox (
  A1 IN NUMBER, A2 IN NUMBER, A3 IN NUMBER, A4 IN NUMBER,
  OBJ    IN FUZZY_LABEL_DEF.OBJ#%TYPE, COL IN FUZZY_LABEL_DEF.OBJ#%TYPE,
  F_TYPE IN FUZZY_COL_LIST.F_TYPE%TYPE,
  F1 IN NUMBER, F2 IN NUMBER, F3 IN NUMBER, F4 IN NUMBER,
  margen IN FUZZY_APPROX_MUCH.MARGEN%TYPE,
  much   IN FUZZY_APPROX_MUCH.MUCH%TYPE) RETURN NUMBER;
PRAGMA RESTRICT_REFERENCES (MLT_aprox,WNDS,WNPS,RNPS);

FUNCTION MLT_trape (
  T1 IN NUMBER, T2 IN NUMBER, T3 IN NUMBER, T4 IN NUMBER,
  OBJ    IN FUZZY_LABEL_DEF.OBJ#%TYPE, COL IN FUZZY_LABEL_DEF.OBJ#%TYPE,
  F_TYPE IN FUZZY_COL_LIST.F_TYPE%TYPE,
  F1 IN NUMBER, F2 IN NUMBER, F3 IN NUMBER, F4 IN NUMBER,
  margen IN FUZZY_APPROX_MUCH.MARGEN%TYPE,
  much   IN FUZZY_APPROX_MUCH.MUCH%TYPE) RETURN NUMBER;
PRAGMA RESTRICT_REFERENCES (MLT_trape,WNDS,WNPS,RNPS);

FUNCTION MLT (
  OBJ1    IN FUZZY_LABEL_DEF.OBJ#%TYPE, COL1 IN FUZZY_LABEL_DEF.OBJ#%TYPE,
  F_TYPE1 IN FUZZY_COL_LIST.F_TYPE%TYPE,
  F11 IN NUMBER, F21 IN NUMBER, F31 IN NUMBER, F41 IN NUMBER,
  OBJ2    IN FUZZY_LABEL_DEF.OBJ#%TYPE, COL2 IN FUZZY_LABEL_DEF.OBJ#%TYPE,
  F_TYPE2 IN FUZZY_COL_LIST.F_TYPE%TYPE,
  F12 IN NUMBER, F22 IN NUMBER, F32 IN NUMBER ,F42 IN NUMBER,
  margen IN FUZZY_APPROX_MUCH.MARGEN%TYPE,
  much   IN FUZZY_APPROX_MUCH.MUCH%TYPE) RETURN NUMBER;
PRAGMA RESTRICT_REFERENCES (MLT,WNDS,WNPS,RNPS);


--******* Comparador difuso NMGT (Necesariamente Mucho Mayor Que) *******--
FUNCTION NMGT_crisp_finter (crisp IN NUMBER, X IN NUMBER, Y IN NUMBER,
  margen IN FUZZY_APPROX_MUCH.MARGEN%TYPE,
  much   IN FUZZY_APPROX_MUCH.MUCH%TYPE) RETURN NUMBER;
PRAGMA RESTRICT_REFERENCES (NMGT_crisp_finter,WNDS,WNPS,RNPS);

FUNCTION NMGT_crisp_aprox (
  crisp IN NUMBER, F1 IN NUMBER, F2 IN NUMBER, F3 IN NUMBER, F4 IN NUMBER,
  margen IN FUZZY_APPROX_MUCH.MARGEN%TYPE,
  much   IN FUZZY_APPROX_MUCH.MUCH%TYPE) RETURN NUMBER;
PRAGMA RESTRICT_REFERENCES (NMGT_crisp_aprox,WNDS,WNPS,RNPS);

FUNCTION NMGT_crisp_trape (
  crisp  IN NUMBER, F1 IN NUMBER, F2 IN NUMBER, F3 IN NUMBER ,F4 IN NUMBER,
  margen IN FUZZY_APPROX_MUCH.MARGEN%TYPE,
  much   IN FUZZY_APPROX_MUCH.MUCH%TYPE) RETURN NUMBER;
PRAGMA RESTRICT_REFERENCES (NMGT_crisp_trape,WNDS,WNPS,RNPS);

FUNCTION NMGT_t1_t1 (
  crisp1 IN NUMBER, crisp2 IN NUMBER,
  margen IN FUZZY_APPROX_MUCH.MARGEN%TYPE,
  much   IN FUZZY_APPROX_MUCH.MUCH%TYPE) RETURN NUMBER;
PRAGMA RESTRICT_REFERENCES (NMGT_t1_t1,WNDS,WNPS,RNPS);

FUNCTION NMGT_t1_t2 (
  crisp  IN NUMBER,
  OBJ IN FUZZY_LABEL_DEF.OBJ#%TYPE, COL IN FUZZY_LABEL_DEF.OBJ#%TYPE,
  F_TYPE IN FUZZY_COL_LIST.F_TYPE%TYPE,
  F1 IN NUMBER, F2 IN NUMBER, F3 IN NUMBER ,F4 IN NUMBER,
  margen IN FUZZY_APPROX_MUCH.MARGEN%TYPE,
  much   IN FUZZY_APPROX_MUCH.MUCH%TYPE) RETURN NUMBER;
PRAGMA RESTRICT_REFERENCES (NMGT_t1_t2,WNDS,WNPS,RNPS);

FUNCTION NMGT_crisp (
  crisp  IN NUMBER,
  OBJ    IN FUZZY_LABEL_DEF.OBJ#%TYPE, COL IN FUZZY_LABEL_DEF.OBJ#%TYPE,
  F_TYPE IN FUZZY_COL_LIST.F_TYPE%TYPE,
  F1 IN NUMBER, F2 IN NUMBER, F3 IN NUMBER ,F4 IN NUMBER,
  margen IN FUZZY_APPROX_MUCH.MARGEN%TYPE,
  much   IN FUZZY_APPROX_MUCH.MUCH%TYPE) RETURN NUMBER;
PRAGMA RESTRICT_REFERENCES (NMGT_crisp,WNDS,WNPS,RNPS);

FUNCTION NMGT_inter (
  X IN NUMBER, Y IN NUMBER,
  OBJ    IN FUZZY_LABEL_DEF.OBJ#%TYPE, COL IN FUZZY_LABEL_DEF.OBJ#%TYPE,
  F_TYPE IN FUZZY_COL_LIST.F_TYPE%TYPE,
  F1 IN NUMBER, F2 IN NUMBER, F3 IN NUMBER, F4 IN NUMBER,
  margen IN FUZZY_APPROX_MUCH.MARGEN%TYPE,
  much   IN FUZZY_APPROX_MUCH.MUCH%TYPE) RETURN NUMBER;
PRAGMA RESTRICT_REFERENCES (NMGT_inter,WNDS,WNPS,RNPS);

FUNCTION NMGT_aprox (
  A1 IN NUMBER, A2 IN NUMBER, A3 IN NUMBER, A4 IN NUMBER,
  OBJ    IN FUZZY_LABEL_DEF.OBJ#%TYPE, COL IN FUZZY_LABEL_DEF.OBJ#%TYPE,
  F_TYPE IN FUZZY_COL_LIST.F_TYPE%TYPE,
  F1 IN NUMBER, F2 IN NUMBER, F3 IN NUMBER, F4 IN NUMBER,
  margen IN FUZZY_APPROX_MUCH.MARGEN%TYPE,
  much   IN FUZZY_APPROX_MUCH.MUCH%TYPE) RETURN NUMBER;
PRAGMA RESTRICT_REFERENCES (NMGT_aprox,WNDS,WNPS,RNPS);

FUNCTION NMGT_trape (
  T1 IN NUMBER, T2 IN NUMBER, T3 IN NUMBER, T4 IN NUMBER,
  OBJ    IN FUZZY_LABEL_DEF.OBJ#%TYPE, COL IN FUZZY_LABEL_DEF.OBJ#%TYPE,
  F_TYPE IN FUZZY_COL_LIST.F_TYPE%TYPE,
  F1 IN NUMBER, F2 IN NUMBER, F3 IN NUMBER, F4 IN NUMBER,
  margen IN FUZZY_APPROX_MUCH.MARGEN%TYPE,
  much   IN FUZZY_APPROX_MUCH.MUCH%TYPE) RETURN NUMBER;
PRAGMA RESTRICT_REFERENCES (NMGT_trape,WNDS,WNPS,RNPS);

FUNCTION NMGT (
  OBJ1    IN FUZZY_LABEL_DEF.OBJ#%TYPE, COL1 IN FUZZY_LABEL_DEF.OBJ#%TYPE,
  F_TYPE1 IN FUZZY_COL_LIST.F_TYPE%TYPE,
  F11 IN NUMBER, F21 IN NUMBER, F31 IN NUMBER, F41 IN NUMBER,
  OBJ2    IN FUZZY_LABEL_DEF.OBJ#%TYPE, COL2 IN FUZZY_LABEL_DEF.OBJ#%TYPE,
  F_TYPE2 IN FUZZY_COL_LIST.F_TYPE%TYPE,
  F12 IN NUMBER, F22 IN NUMBER, F32 IN NUMBER ,F42 IN NUMBER,
  margen IN FUZZY_APPROX_MUCH.MARGEN%TYPE,
  much   IN FUZZY_APPROX_MUCH.MUCH%TYPE) RETURN NUMBER;
PRAGMA RESTRICT_REFERENCES (NMGT,WNDS,WNPS,RNPS);

--******* Comparador difuso NMLT (Necesariamente Mucho Menor Que) *******--
FUNCTION NMLT_crisp_finter (crisp  IN NUMBER, X IN NUMBER, Y IN NUMBER,
  margen IN FUZZY_APPROX_MUCH.MARGEN%TYPE,
  much   IN FUZZY_APPROX_MUCH.MUCH%TYPE) RETURN NUMBER;
PRAGMA RESTRICT_REFERENCES (NMLT_crisp_finter,WNDS,WNPS,RNPS);

FUNCTION NMLT_crisp_aprox (
  crisp  IN NUMBER, F1 IN NUMBER, F2 IN NUMBER, F3 IN NUMBER, F4 IN NUMBER,
  margen IN FUZZY_APPROX_MUCH.MARGEN%TYPE,
  much   IN FUZZY_APPROX_MUCH.MUCH%TYPE) RETURN NUMBER;
PRAGMA RESTRICT_REFERENCES (NMLT_crisp_aprox,WNDS,WNPS,RNPS);

FUNCTION NMLT_crisp_trape (
  crisp  IN NUMBER, F1 IN NUMBER, F2 IN NUMBER, F3 IN NUMBER ,F4 IN NUMBER,
  margen IN FUZZY_APPROX_MUCH.MARGEN%TYPE,
  much   IN FUZZY_APPROX_MUCH.MUCH%TYPE) RETURN NUMBER;
PRAGMA RESTRICT_REFERENCES (NMLT_crisp_trape,WNDS,WNPS,RNPS);

FUNCTION NMLT_t1_t1 (
  crisp1 IN NUMBER, crisp2 IN NUMBER,
  margen IN FUZZY_APPROX_MUCH.MARGEN%TYPE,
  much   IN FUZZY_APPROX_MUCH.MUCH%TYPE) RETURN NUMBER;
PRAGMA RESTRICT_REFERENCES (NMLT_t1_t1,WNDS,WNPS,RNPS);

FUNCTION NMLT_t1_t2 (
  crisp  IN NUMBER,
  OBJ    IN FUZZY_LABEL_DEF.OBJ#%TYPE, COL IN FUZZY_LABEL_DEF.OBJ#%TYPE,
  F_TYPE IN FUZZY_COL_LIST.F_TYPE%TYPE,
  F1 IN NUMBER, F2 IN NUMBER, F3 IN NUMBER ,F4 IN NUMBER,
  margen IN FUZZY_APPROX_MUCH.MARGEN%TYPE,
  much   IN FUZZY_APPROX_MUCH.MUCH%TYPE) RETURN NUMBER;
PRAGMA RESTRICT_REFERENCES (NMLT_t1_t2,WNDS,WNPS,RNPS);

FUNCTION NMLT_crisp (
  crisp  IN NUMBER,
  OBJ    IN FUZZY_LABEL_DEF.OBJ#%TYPE, COL IN FUZZY_LABEL_DEF.OBJ#%TYPE,
  F_TYPE IN FUZZY_COL_LIST.F_TYPE%TYPE,
  F1 IN NUMBER, F2 IN NUMBER, F3 IN NUMBER ,F4 IN NUMBER,
  margen IN FUZZY_APPROX_MUCH.MARGEN%TYPE,
  much   IN FUZZY_APPROX_MUCH.MUCH%TYPE) RETURN NUMBER;
PRAGMA RESTRICT_REFERENCES (NMLT_crisp,WNDS,WNPS,RNPS);

FUNCTION NMLT_inter (
  X IN NUMBER, Y IN NUMBER,
  OBJ    IN FUZZY_LABEL_DEF.OBJ#%TYPE, COL IN FUZZY_LABEL_DEF.OBJ#%TYPE,
  F_TYPE IN FUZZY_COL_LIST.F_TYPE%TYPE,
  F1 IN NUMBER, F2 IN NUMBER, F3 IN NUMBER, F4 IN NUMBER,
  margen IN FUZZY_APPROX_MUCH.MARGEN%TYPE,
  much   IN FUZZY_APPROX_MUCH.MUCH%TYPE) RETURN NUMBER;
PRAGMA RESTRICT_REFERENCES (NMLT_inter,WNDS,WNPS,RNPS);

FUNCTION NMLT_aprox (
  A1 IN NUMBER, A2 IN NUMBER, A3 IN NUMBER, A4 IN NUMBER,
  OBJ    IN FUZZY_LABEL_DEF.OBJ#%TYPE, COL IN FUZZY_LABEL_DEF.OBJ#%TYPE,
  F_TYPE IN FUZZY_COL_LIST.F_TYPE%TYPE,
  F1 IN NUMBER, F2 IN NUMBER, F3 IN NUMBER, F4 IN NUMBER,
  margen IN FUZZY_APPROX_MUCH.MARGEN%TYPE,
  much   IN FUZZY_APPROX_MUCH.MUCH%TYPE) RETURN NUMBER;
PRAGMA RESTRICT_REFERENCES (NMLT_aprox,WNDS,WNPS,RNPS);

FUNCTION NMLT_trape (
  T1 IN NUMBER, T2 IN NUMBER, T3 IN NUMBER, T4 IN NUMBER,
  OBJ    IN FUZZY_LABEL_DEF.OBJ#%TYPE, COL IN FUZZY_LABEL_DEF.OBJ#%TYPE,
  F_TYPE IN FUZZY_COL_LIST.F_TYPE%TYPE,
  F1 IN NUMBER, F2 IN NUMBER, F3 IN NUMBER, F4 IN NUMBER,
  margen IN FUZZY_APPROX_MUCH.MARGEN%TYPE,
  much   IN FUZZY_APPROX_MUCH.MUCH%TYPE) RETURN NUMBER;
PRAGMA RESTRICT_REFERENCES (NMLT_trape,WNDS,WNPS,RNPS);

FUNCTION NMLT (
  OBJ1    IN FUZZY_LABEL_DEF.OBJ#%TYPE, COL1 IN FUZZY_LABEL_DEF.OBJ#%TYPE,
  F_TYPE1 IN FUZZY_COL_LIST.F_TYPE%TYPE,
  F11 IN NUMBER, F21 IN NUMBER, F31 IN NUMBER, F41 IN NUMBER,
  OBJ2    IN FUZZY_LABEL_DEF.OBJ#%TYPE, COL2 IN FUZZY_LABEL_DEF.OBJ#%TYPE,
  F_TYPE2 IN FUZZY_COL_LIST.F_TYPE%TYPE,
  F12 IN NUMBER, F22 IN NUMBER, F32 IN NUMBER ,F42 IN NUMBER,
  margen IN FUZZY_APPROX_MUCH.MARGEN%TYPE,
  much   IN FUZZY_APPROX_MUCH.MUCH%TYPE) RETURN NUMBER;
PRAGMA RESTRICT_REFERENCES (NMLT,WNDS,WNPS,RNPS);

END FSQL_FUNCTIONS2;
/

CREATE OR REPLACE PACKAGE BODY FSQL_FUNCTIONS2 AS

--********************************************************************
-- Comparacin FGT, FLT, FGEQ, FLEQ, MGT y MLT de cualquier valor
-- difuso tipo 2 con las ctes difusas UNKNOWN, UNDEFINED y NULL.
--********************************************************************

----------------------------------------------------------------------
-- Indica si un atributo difuso tipo 2 es Mayor o Menor que la cte. difusa UNKNOWN
----------------------------------------------------------------------
FUNCTION MayorMenor_UNKNOWN (F_TYPE IN FUZZY_COL_LIST.F_TYPE%TYPE) RETURN NUMBER IS
BEGIN
  -- UNKNOWN es 'mayor' (y tambin 'menor') que cualquier cosa, excepto que UNDEFINED
  IF F_TYPE=1 THEN RETURN 0; -- Undefined>Unknown? NO
  ELSE             RETURN 1;
  END IF;
END MayorMenor_UNKNOWN;

----------------------------------------------------------------------
-- Indica si un atributo difuso tipo 2 es Mayor o Menor que la cte. difusa UNDEFINED
----------------------------------------------------------------------
FUNCTION MayorMenor_UNDEFINED RETURN NUMBER IS
BEGIN
  RETURN 0;
END MayorMenor_UNDEFINED;

----------------------------------------------------------------------
-- Indica si un atributo difuso tipo 2 es Mayor o Menor que la cte. difusa NULL
----------------------------------------------------------------------
FUNCTION MayorMenor_NULL (F_TYPE IN FUZZY_COL_LIST.F_TYPE%TYPE) RETURN NUMBER IS
BEGIN
  -- NULL es 'mayor' (y tambin 'menor') que cualquier cosa, excepto que UNDEFINED
  IF F_TYPE=1 THEN RETURN 0; -- Undefined>Null? NO
  ELSE             RETURN 1;
  END IF;
END MayorMenor_NULL;

--*********** Comparador difuso MGT  (Much Greater Than) ***********--
-- Para ver si un valor A es 'MUCHO MAYOR' que otro B, se le suma a B
-- la constante much (que depende de cada atributo y est almacenada en
-- la tabla FUZZY_MARGEN_MUCH), y se compara como el comparador FGT.
-- Adems, con los comparadores MGT y MLT se difuminan los valores no difusos:
-- Los valores crisp se convierten en un valor aprox y los valores
-- intervalo se convierten en un trapecio:
--	Crisp n          ==> #n (nmargen)
--	Intervalo [x,y]  ==> $[x-margen,x,y,y+margen]
-- Esta conversin se realiza porque las comparadores 'MUCHO...' son difusos
-- por s mismos, por lo que no tratan valores crisp.
-- Por esta ltima conversin, no se pueden usar las funciones de comparacin
-- de los comparadores FGT y FLT.

----------------------------------------------------------------------
-- Indica si un valor crisp es 'MUCHO MAYOR' (MGT) que un intervalo [X,Y]
-- El valor crisp se difumina a un APROX y el intervalo a un trapecio.
-- Caso ejemplo: fcol_t1 MGT [3,8]
----------------------------------------------------------------------
FUNCTION MGT_crisp_finter (
  crisp  IN NUMBER, X IN NUMBER, Y IN NUMBER,
  margen IN FUZZY_APPROX_MUCH.MARGEN%TYPE,
  much   IN FUZZY_APPROX_MUCH.MUCH%TYPE) RETURN NUMBER IS
BEGIN
  -- Comparacin: #crisp MGT $[X+much-margen,X+much,Y+much,Y+much+margen]
  IF    crisp>=Y+much+margen THEN RETURN 1;
  ELSIF crisp+margen>Y+much  THEN
        RETURN ROUND( (crisp+margen-Y-much)/(margen+margen) ,2);
  ELSE  RETURN 0;
  END IF;
END MGT_crisp_finter;

----------------------------------------------------------------------
-- Indica si un valor crisp es 'MUCHO MAYOR' (MGT) que un valor aprox
-- Caso ejemplo: fcol_t1 FGT #6
-- Valor aprox:  F1  F4 (donde F4 es el margen: F1-F4=F2 y F1+F4=F3)
----------------------------------------------------------------------
FUNCTION MGT_crisp_aprox (
  crisp  IN NUMBER,
  F1 IN NUMBER, F2 IN NUMBER, F3 IN NUMBER, F4 IN NUMBER,
  margen IN FUZZY_APPROX_MUCH.MARGEN%TYPE,
  much   IN FUZZY_APPROX_MUCH.MUCH%TYPE) RETURN NUMBER IS
BEGIN
  -- Comparacin: crispmargen FGT (F1+much)F4
  IF    crisp>=F3+much THEN RETURN 1;
  ELSIF crisp+margen>F1+much THEN
    RETURN ROUND( (crisp+margen-F1-much)/(F4+margen) ,2);
  ELSE  RETURN 0;
  END IF;
END MGT_crisp_aprox;

----------------------------------------------------------------------
-- Indica si un valor crisp es 'MUCHO MAYOR' (MGT) que un trapecio
-- Caso ejemplo: fcol_t1 MGT $[1,4,7,9]
----------------------------------------------------------------------
FUNCTION MGT_crisp_trape (
  crisp  IN NUMBER,
  F1 IN NUMBER, F2 IN NUMBER, F3 IN NUMBER ,F4 IN NUMBER,
  margen IN FUZZY_APPROX_MUCH.MARGEN%TYPE,
  much   IN FUZZY_APPROX_MUCH.MUCH%TYPE) RETURN NUMBER IS
BEGIN
  -- alfa=F1, delta=F4, beta:=F2+F1; gamma:=F3+F4;
  IF    crisp>=F4+much THEN RETURN 1;
  ELSIF crisp+margen>F3+F4+much THEN
    RETURN ROUND( (crisp+margen-F3-F4-much)/(margen-F3) ,2);
  ELSE  RETURN 0;
  END IF;
END MGT_crisp_trape;

----------------------------------------------------------------------
-- Indica si un valor crisp es 'MUCHO MAYOR' (MGT) que una etiqueta.
-- No se exporta (caso fcol_t1 MGT $label se traduce a MGT_crisp_trape)
----------------------------------------------------------------------
FUNCTION MGT_crisp_label (
  crisp  IN NUMBER,
  OBJ    IN FUZZY_LABEL_DEF.OBJ#%TYPE, COL IN FUZZY_LABEL_DEF.OBJ#%TYPE,
  F_ID   IN FUZZY_LABEL_DEF.FUZZY_ID%TYPE,
  margen IN FUZZY_APPROX_MUCH.MARGEN%TYPE,
  much   IN FUZZY_APPROX_MUCH.MUCH%TYPE) RETURN NUMBER
IS
  gamma  FUZZY_LABEL_DEF.ALFA%TYPE;
  delta  FUZZY_LABEL_DEF.ALFA%TYPE;
BEGIN
  SELECT GAMMA,DELTA INTO gamma,delta FROM FUZZY_LABEL_DEF
    WHERE OBJ#=OBJ AND COL#=COL AND FUZZY_ID=F_ID;
  IF    crisp>=delta+much THEN RETURN 1;
  ELSIF crisp+margen>gamma+much THEN
    RETURN ROUND( (crisp+margen-gamma-much)/(delta-gamma+margen) ,2);
  ELSE  RETURN 0;
  END IF;
END MGT_crisp_label;

----------------------------------------------------------------------
-- Indica si una columna difusa tipo 1 (crisp1) es 'MUCHO MAYOR' (FGT)
-- que un valor crisp, otra columna tipo 1 o una expresin crisp (crisp2).
-- Formatos: fcol_t1 MGT fcol_t1, fcol_t1 MGT crisp2  fcol_t1 MGT expr_crisp
-- Este comparador es difuso 'per se' (no difuminado a partir de uno crisp)
-- por tanto sus comparaciones deben ser ms difuminadas: Por ello, se difuminan
-- los 2 valores crisp a comparar suponiendo que ambos crisp son valores aprox.
----------------------------------------------------------------------
FUNCTION MGT_t1_t1 (
  crisp1 IN NUMBER, crisp2 IN NUMBER,
  margen IN FUZZY_APPROX_MUCH.MARGEN%TYPE,
  much   IN FUZZY_APPROX_MUCH.MUCH%TYPE) RETURN NUMBER IS
BEGIN
  IF crisp1>=crisp2+margen+much THEN RETURN 1;
  ELSIF crisp1+margen>crisp2+much THEN
       RETURN ROUND( (crisp1+margen-crisp2-much)/(margen+margen) ,2);
  ELSE RETURN 0;
  END IF;
END MGT_t1_t1;

----------------------------------------------------------------------
-- Indica si una columna difusa tipo 1 (crisp) es 'MUCHO MAYOR' (MGT)
-- que una columna tipo 2. Formato: fcol_t1 MGT fcol_t2
----------------------------------------------------------------------
FUNCTION MGT_t1_t2 (
  crisp  IN NUMBER,
  OBJ    IN FUZZY_LABEL_DEF.OBJ#%TYPE, COL IN FUZZY_LABEL_DEF.OBJ#%TYPE,
  F_TYPE IN FUZZY_COL_LIST.F_TYPE%TYPE,
  F1 IN NUMBER, F2 IN NUMBER, F3 IN NUMBER ,F4 IN NUMBER,
  margen IN FUZZY_APPROX_MUCH.MARGEN%TYPE,
  much   IN FUZZY_APPROX_MUCH.MUCH%TYPE) RETURN NUMBER IS
BEGIN
  IF    F_TYPE=0 THEN RETURN MayorMenor_UNKNOWN(3); -- 3 es el tipo crisp
  ELSIF F_TYPE=1 THEN RETURN MayorMenor_UNDEFINED;
  ELSIF F_TYPE=2 THEN RETURN MayorMenor_NULL(3);
  ELSIF F_TYPE=3 THEN RETURN MGT_t1_t1(crisp,F1,margen,much);
  ELSIF F_TYPE=4 THEN RETURN MGT_crisp_label(crisp,OBJ,COL,F1,margen,much);
  ELSIF F_TYPE=5 THEN RETURN MGT_crisp_finter(crisp,F1,F4,margen,much);
  ELSIF F_TYPE=6 THEN RETURN MGT_crisp_aprox(crisp,F1,F2,F3,F4,margen,much);
  ELSE                RETURN MGT_crisp_trape(crisp,F1,F2,F3,F4,margen,much);
  END IF;
END MGT_t1_t2;

--********************************************************************
-- Indica si una columna difusa tipo 2 es 'MUCHO MAYOR' (MGT) que un crisp
-- Caso ejemplo: fcol_t2 MGT 8. Tambin: fcol_t2 MGT fcol_t1
--********************************************************************
FUNCTION MGT_crisp (
  crisp  IN NUMBER,
  OBJ    IN FUZZY_LABEL_DEF.OBJ#%TYPE, COL IN FUZZY_LABEL_DEF.OBJ#%TYPE,
  F_TYPE IN FUZZY_COL_LIST.F_TYPE%TYPE,
  F1 IN NUMBER, F2 IN NUMBER, F3 IN NUMBER ,F4 IN NUMBER,
  margen IN FUZZY_APPROX_MUCH.MARGEN%TYPE,
  much   IN FUZZY_APPROX_MUCH.MUCH%TYPE) RETURN NUMBER
IS
  gamma FUZZY_LABEL_DEF.ALFA%TYPE;
  delta FUZZY_LABEL_DEF.ALFA%TYPE;
BEGIN
  IF    F_TYPE=0 THEN RETURN MayorMenor_UNKNOWN(3); -- 3 es el tipo crisp
  ELSIF F_TYPE=1 THEN RETURN MayorMenor_UNDEFINED;
  ELSIF F_TYPE=2 THEN RETURN MayorMenor_NULL(3);
  ELSIF F_TYPE=3 THEN RETURN MGT_t1_t1(F1,crisp,margen,much);
  ELSIF F_TYPE=4 THEN -- $label MGT #crisp
     SELECT GAMMA,DELTA INTO gamma,delta FROM FUZZY_LABEL_DEF
       WHERE OBJ#=OBJ AND COL#=COL AND FUZZY_ID=F1;
     IF    gamma>=crisp+margen+much THEN RETURN 1;
     ELSIF delta> crisp+much THEN RETURN ROUND( (delta-crisp-much)/(margen-gamma+delta) ,2);
     ELSE  RETURN 0;
     END IF;
  ELSIF F_TYPE=5 THEN -- Comparacin $[F1-margen,F1,F4,F4+margen] MGT #crisp
     IF    F4>=crisp+margen+much THEN RETURN 1;
     ELSIF F4+margen>crisp+much THEN RETURN ROUND( (F4+margen-crisp-much)/(margen+margen) ,2);
     ELSE  RETURN 0;
     END IF;
  ELSIF F_TYPE=6 THEN -- Comparacin #F1 MGT #crisp
     IF    F1>=crisp+margen+much THEN RETURN 1;
     ELSIF F1+margen>crisp+much THEN RETURN ROUND( (F3-crisp-much)/(margen+F4) ,2);
     ELSE  RETURN 0;
     END IF;
  ELSE  -- Comparacin $[F1,F2+F1,F3+F4,F4] MGT #crisp
     IF    F3+F4>=crisp+margen+much THEN RETURN 1;
     ELSIF F4>crisp+much THEN RETURN ROUND( (F4-crisp-much)/(margen-F3) ,2);
     ELSE  RETURN 0;
     END IF;
  END IF;

END MGT_crisp;

--********************************************************************
-- Indica si una columna difusa tipo 2 es 'MUCHO MAYOR' (MGT) que un intervalo [X,Y]
-- Caso ejemplo: fcol_t2 MGT [X,Y]
--********************************************************************
FUNCTION MGT_inter (
  X   IN NUMBER, Y IN NUMBER,
  OBJ IN FUZZY_LABEL_DEF.OBJ#%TYPE, COL IN FUZZY_LABEL_DEF.OBJ#%TYPE,
  F_TYPE IN FUZZY_COL_LIST.F_TYPE%TYPE,
  F1 IN NUMBER, F2 IN NUMBER, F3 IN NUMBER, F4 IN NUMBER,
  margen IN FUZZY_APPROX_MUCH.MARGEN%TYPE,
  much   IN FUZZY_APPROX_MUCH.MUCH%TYPE) RETURN NUMBER
IS
  gamma FUZZY_LABEL_DEF.ALFA%TYPE;
  delta FUZZY_LABEL_DEF.ALFA%TYPE;
BEGIN
  IF    F_TYPE=0 THEN RETURN MayorMenor_UNKNOWN(5); -- 5 es el tipo intervalo
  ELSIF F_TYPE=1 THEN RETURN MayorMenor_UNDEFINED;
  ELSIF F_TYPE=2 THEN RETURN MayorMenor_NULL(5);
  ELSIF F_TYPE=3 THEN RETURN MGT_crisp_finter(F1,X,Y,margen,much);
  ELSIF F_TYPE=4 THEN -- $label MGT $[X-margen+much,X+much,Y+much,Y+margen+much]
    SELECT GAMMA,DELTA INTO gamma,delta FROM FUZZY_LABEL_DEF
      WHERE OBJ#=OBJ AND COL#=COL AND FUZZY_ID=F1;
    IF    gamma>=Y+margen+much THEN RETURN 1;
    ELSIF delta> Y+much THEN RETURN ROUND( (delta-Y-much)/(margen-gamma+delta) ,2);
    ELSE  RETURN 0;
    END IF;
  ELSIF F_TYPE=5 THEN -- $[F1-margen,F1,F4,F4+margen] MGT $[X-margen+much,X+much,Y+much,Y+margen+much]
    IF    F4>=Y+margen+much THEN RETURN 1;
    ELSIF F1+margen>Y+much THEN RETURN ROUND( (F4+margen-Y-much)/(margen+margen) ,2);
    ELSE  RETURN 0;
    END IF;
  ELSIF F_TYPE=6 THEN -- #F1 FGT $[X-margen+much,X+much,Y+much,Y+margen+much]
    IF    F1>=Y+margen+much THEN RETURN 1;
    ELSIF F3> Y+much THEN RETURN ROUND( (F3-Y-much)/(margen+F4) ,2);
    ELSE  RETURN 0;
    END IF;
  ELSE -- F_TYPE=7: $[F1,F2+F1,F3+F4,F4] FGT $[X-margen+much,X+much,Y+much,Y+margen+much]
    IF    F3+F4>=Y+margen+much THEN RETURN 1;
    ELSIF F4>Y+much THEN RETURN ROUND( (F4-Y-much)/(margen-F3) ,2);
    ELSE  RETURN 0;
    END IF;
  END IF;
END MGT_inter;

--********************************************************************
-- Indica si una columna difusa tipo 2 es 'MUCHO MAYOR' (MGT) que un aprox [A1,A2,A3,A4]
-- Caso ejemplo: fcol_t2 MGT #A1
--********************************************************************
FUNCTION MGT_aprox (
  A1  IN NUMBER, A2 IN NUMBER, A3 IN NUMBER, A4 IN NUMBER,
  OBJ IN FUZZY_LABEL_DEF.OBJ#%TYPE, COL IN FUZZY_LABEL_DEF.OBJ#%TYPE,
  F_TYPE IN FUZZY_COL_LIST.F_TYPE%TYPE,
  F1 IN NUMBER, F2 IN NUMBER, F3 IN NUMBER, F4 IN NUMBER,
  margen IN FUZZY_APPROX_MUCH.MARGEN%TYPE,
  much   IN FUZZY_APPROX_MUCH.MUCH%TYPE) RETURN NUMBER
IS
  gamma FUZZY_LABEL_DEF.ALFA%TYPE;
  delta FUZZY_LABEL_DEF.ALFA%TYPE;
BEGIN
  IF    F_TYPE=0 THEN RETURN MayorMenor_UNKNOWN(6); -- 6 es el tipo aprox
  ELSIF F_TYPE=1 THEN RETURN MayorMenor_UNDEFINED;
  ELSIF F_TYPE=2 THEN RETURN MayorMenor_NULL(6);
  ELSIF F_TYPE=3 THEN RETURN MGT_crisp_aprox(F1,A1,A2,A3,A4,margen,much);
  ELSIF F_TYPE=4 THEN -- $label MGT #A1+much
    SELECT GAMMA,DELTA INTO gamma,delta FROM FUZZY_LABEL_DEF
      WHERE OBJ#=OBJ AND COL#=COL AND FUZZY_ID=F1;
    IF    gamma>=A3+much THEN RETURN 1;
    ELSIF delta> A1+much THEN RETURN ROUND( (delta-A1-much)/(A4-gamma+delta) ,2);
    ELSE  RETURN 0;
    END IF;
  ELSIF F_TYPE=5 THEN -- $[F1-margen,F1,F4,F4+margen] FGT #A1+much
   IF    F4>=A3+much THEN RETURN 1;
   ELSIF F4+margen>A1+much THEN RETURN ROUND( (F4+margen-A1-much)/(A4+margen) ,2);
   ELSE  RETURN 0;
   END IF;
  ELSIF F_TYPE=6 THEN -- #F1 FGT #A1+much
    IF    F1>=A3+much THEN RETURN 1;
    ELSIF F3> A1+much THEN RETURN ROUND( (F3-A1-much)/(A4+F4) ,2);
    ELSE  RETURN 0;
    END IF;
  ELSE -- F_TYPE=7: $[F1,F2+F1,F3+F4,F4] FGT #A1+much
    IF    F3+F4>=A3+much THEN RETURN 1;
    ELSIF F4   > A1+much THEN RETURN ROUND( (F4-A1-much)/(A4-F3) ,2);
    ELSE  RETURN 0;
    END IF;
  END IF;
END MGT_aprox;

--********************************************************************
-- Indica si una columna difusa tipo 2 es 'MUCHO MAYOR' (MGT) que un trapecio
-- Caso ejemplo: fcol_t2 MGT $[T1,T2,T3,T4]
--********************************************************************
FUNCTION MGT_trape (
  T1 IN NUMBER, T2 IN NUMBER, T3 IN NUMBER, T4 IN NUMBER,
  OBJ    IN FUZZY_LABEL_DEF.OBJ#%TYPE, COL IN FUZZY_LABEL_DEF.OBJ#%TYPE,
  F_TYPE IN FUZZY_COL_LIST.F_TYPE%TYPE,
  F1 IN NUMBER, F2 IN NUMBER, F3 IN NUMBER, F4 IN NUMBER,
  margen IN FUZZY_APPROX_MUCH.MARGEN%TYPE,
  much   IN FUZZY_APPROX_MUCH.MUCH%TYPE) RETURN NUMBER
IS
  gamma FUZZY_LABEL_DEF.ALFA%TYPE;
  delta FUZZY_LABEL_DEF.ALFA%TYPE;
BEGIN
  -- Trapecio: alfa=T1, delta=T4, beta:=T2+T1; gamma:=T3+T4;
  IF    F_TYPE=0 THEN RETURN MayorMenor_UNKNOWN(7); -- 7 es el tipo trapecio
  ELSIF F_TYPE=1 THEN RETURN MayorMenor_UNDEFINED;
  ELSIF F_TYPE=2 THEN RETURN MayorMenor_NULL(7);
  ELSIF F_TYPE=3 THEN RETURN MGT_crisp_trape(F1,T1,T2,T3,T4,margen,much);
  ELSIF F_TYPE=4 THEN -- $label FGT $[T1+much,T2+T1+much,T3+T4+much,T4+much]
    SELECT GAMMA,DELTA INTO gamma,delta FROM FUZZY_LABEL_DEF
      WHERE OBJ#=OBJ AND COL#=COL AND FUZZY_ID=F1;
    IF    gamma>=T4+much    THEN RETURN 1;
    ELSIF delta> T3+T4+much THEN RETURN ROUND( (delta-T3-T4-much)/(delta-gamma-T3) ,2);
    ELSE  RETURN 0;
    END IF;
  ELSIF F_TYPE=5 THEN -- $[F1-margen,F1,F4,F4+margen] FGT $[T1,T2+T1,T3+T4,T4]+much
    IF    F4       >=T4+much    THEN RETURN 1;
    ELSIF F4+margen> T3+T4+much THEN RETURN ROUND( (F4+margen-T3-T4-much)/(margen-T3) ,2);
    ELSE  RETURN 0;
    END IF;
  ELSIF F_TYPE=6 THEN -- #F1 FGT $[T1,T2+T1,T3+T4,T4]+much
    IF    F1>=T4+much    THEN RETURN 1;
    ELSIF F3> T3+T4+much THEN RETURN ROUND( (F3-T3-T4-much)/(F4-T3) ,2);
    ELSE  RETURN 0;
    END IF;
  ELSE -- F_TYPE=7: $[F1,F2+F1,F3+F4,F4] FGT $[T1,T2+T1,T3+T4,T4]+much
    IF    F3+F4>=T4+much    THEN RETURN 1;
    ELSIF F4   > T3+T4+much THEN RETURN ROUND( (F4-T3-T4-much)/(-F3-T3) ,2);
    ELSE  RETURN 0;
    END IF;
  END IF;
END MGT_trape;

--********************************************************************
-- Funcin difusa de comparacin: "MUCHO MAYOR QUE": MGT (Much Greater Than).
-- Indica si una columna difusa tipo 2 es 'MUCHO MAYOR QUE' (MGT) otra
-- Caso ejemplo: fcol1_t2 MGT fcol2_t2
--********************************************************************
FUNCTION MGT (
  OBJ1    IN FUZZY_LABEL_DEF.OBJ#%TYPE, COL1 IN FUZZY_LABEL_DEF.OBJ#%TYPE,
  F_TYPE1 IN FUZZY_COL_LIST.F_TYPE%TYPE,
  F11 IN NUMBER, F21 IN NUMBER, F31 IN NUMBER, F41 IN NUMBER,
  OBJ2    IN FUZZY_LABEL_DEF.OBJ#%TYPE, COL2 IN FUZZY_LABEL_DEF.OBJ#%TYPE,
  F_TYPE2 IN FUZZY_COL_LIST.F_TYPE%TYPE,
  F12 IN NUMBER, F22 IN NUMBER, F32 IN NUMBER ,F42 IN NUMBER,
  margen IN FUZZY_APPROX_MUCH.MARGEN%TYPE,
  much   IN FUZZY_APPROX_MUCH.MUCH%TYPE) RETURN NUMBER
IS
  alfa  FUZZY_LABEL_DEF.ALFA%TYPE;  gamma FUZZY_LABEL_DEF.ALFA%TYPE;
  beta  FUZZY_LABEL_DEF.ALFA%TYPE;  delta FUZZY_LABEL_DEF.ALFA%TYPE;
BEGIN
  IF    F_TYPE2=0 THEN RETURN MayorMenor_UNKNOWN(F_TYPE1);
  ELSIF F_TYPE2=1 THEN RETURN MayorMenor_UNDEFINED;
  ELSIF F_TYPE2=2 THEN RETURN MayorMenor_NULL(F_TYPE1);
  ELSIF F_TYPE1=0 THEN RETURN MayorMenor_UNKNOWN(F_TYPE2); -- Para ahorrar llamadas
  ELSIF F_TYPE1=1 THEN RETURN MayorMenor_UNDEFINED;
  ELSIF F_TYPE1=2 THEN RETURN MayorMenor_NULL(F_TYPE2);
  ELSIF F_TYPE2=3 THEN -- CRISP
    RETURN MGT_crisp(F12,OBJ1,COL1,F_TYPE1,F11,F21,F31,F41,margen,much);
  ELSIF F_TYPE2=4 THEN -- LABEL: Calculamos los valores del trapecio
    SELECT ALFA,BETA,GAMMA,DELTA INTO alfa,beta,gamma,delta FROM FUZZY_LABEL_DEF
      WHERE OBJ#=OBJ2 AND COL#=COL2 AND FUZZY_ID=F12;
    RETURN MGT_trape(alfa,beta-alfa,gamma-delta,delta,OBJ1,COL1,F_TYPE1,F11,F21,F31,F41,margen,much);
  ELSIF F_TYPE2=5 THEN -- INTERVALO
    RETURN MGT_inter(F12,F42,OBJ1,COL1,F_TYPE1,F11,F21,F31,F41,margen,much);
  ELSIF F_TYPE2=6 THEN -- APROX
    RETURN MGT_aprox(F12,F22,F32,F42,OBJ1,COL1,F_TYPE1,F11,F21,F31,F41,margen,much);
  ELSE -- F_TYPE2=7: TRAPECIO: alfa=F1, delta=F4, beta:=F2+F1; gamma:=F3+F4;
    RETURN MGT_trape(F12,F22,F32,F42,OBJ1,COL1,F_TYPE1,F11,F21,F31,F41,margen,much);
  END IF;
END MGT;

--*********** Comparador difuso MLT  (Much Less Than) ***********--
-- Para ver si un valor A es 'MUCHO MENOR' que otro B, se le resta a B
-- la constante much (que depende de cada atributo y est almacenada en
-- la tabla FUZZY_MARGEN_MUCH), y se compara como el comparador FLT.
-- Los crisp e intervalo son difuminados a aprox y trapecio respectivamente.

----------------------------------------------------------------------
-- Indica si un valor crisp es 'MUCHO MENOR' (MLT) que un intervalo [X,Y]
-- Caso ejemplo: fcol_t1 MLT [3,8]
----------------------------------------------------------------------
FUNCTION MLT_crisp_finter (
  crisp  IN NUMBER,
  X IN NUMBER, Y IN NUMBER,
  margen IN FUZZY_APPROX_MUCH.MARGEN%TYPE,
  much   IN FUZZY_APPROX_MUCH.MUCH%TYPE) RETURN NUMBER IS
BEGIN -- Se puede hacer con una llamada a MGT_inter (pero es menos eficiente)
  IF    crisp<=X-margen-much THEN RETURN 1;
  ELSIF crisp-margen<X-much  THEN RETURN ROUND( (X-much-crisp+margen)/(margen+margen) ,2);
  ELSE  RETURN 0;
  END IF;
END MLT_crisp_finter;

----------------------------------------------------------------------
-- Indica si un valor crisp es 'MUCHO MENOR' (MLT) que un valor aprox
-- Caso ejemplo: fcol_t1 MLT #6
-- Valor aprox:  F1 + - F4 (donde F4 es el margen: F1-F4=F2 y F1+F4=F3)
----------------------------------------------------------------------
FUNCTION MLT_crisp_aprox (
  crisp  IN NUMBER,
  F1 IN NUMBER, F2 IN NUMBER, F3 IN NUMBER, F4 IN NUMBER,
  margen IN FUZZY_APPROX_MUCH.MARGEN%TYPE,
  much   IN FUZZY_APPROX_MUCH.MUCH%TYPE) RETURN NUMBER IS
BEGIN
  IF    crisp      <=F2-much THEN RETURN 1;
  ELSIF crisp-margen<F1-much THEN RETURN ROUND( (F1-much-crisp+margen)/(F4+margen) ,2);
  ELSE  RETURN 0;
  END IF;
END MLT_crisp_aprox;

----------------------------------------------------------------------
-- Indica si un valor crisp es 'MUCHO MENOR' (MLT) que un trapecio
-- Caso ejemplo: fcol_t1 MLT $[1,4,7,9]
----------------------------------------------------------------------
FUNCTION MLT_crisp_trape (
  crisp  IN NUMBER,
  F1 IN NUMBER, F2 IN NUMBER, F3 IN NUMBER ,F4 IN NUMBER,
  margen IN FUZZY_APPROX_MUCH.MARGEN%TYPE,
  much   IN FUZZY_APPROX_MUCH.MUCH%TYPE) RETURN NUMBER IS
BEGIN -- F_TYPE=7: alfa=F1, delta=F4, beta:=F2+F1; gamma:=F3+F4;
  IF    crisp<=F1-much THEN RETURN 1;
  ELSIF crisp-margen<F1+F2-much THEN
        RETURN ROUND( (F1+F2-much-crisp+margen)/(F2+margen) ,2);
  ELSE  RETURN 0;
  END IF;
END MLT_crisp_trape;

----------------------------------------------------------------------
-- Indica si una columna difusa tipo 1 (crisp1) es 'MUCHO MENOR' (MLT)
-- que un valor crisp u otra columna tipo 1 (crisp2).
-- Formato: fcol_t1 MLT fcol_t1, o tambin  fcol_t1 MLT crisp2
----------------------------------------------------------------------
FUNCTION MLT_t1_t1 (
  crisp1 IN NUMBER, crisp2 IN NUMBER,
  margen IN FUZZY_APPROX_MUCH.MARGEN%TYPE,
  much   IN FUZZY_APPROX_MUCH.MUCH%TYPE) RETURN NUMBER IS
BEGIN
  IF    crisp1<=crisp2-margen-much THEN RETURN 1;
  ELSIF crisp1-margen<crisp2-much THEN
        RETURN ROUND( (crisp2-much-crisp1+margen)/(margen+margen) ,2);
  ELSE  RETURN 0;
  END IF;
END MLT_t1_t1;

----------------------------------------------------------------------
-- Indica si una columna difusa tipo 1 (crisp) es 'MUCHO MENOR' (MLT)
-- que una columna tipo 2. Formato: fcol_t1 MLT fcol_t2
----------------------------------------------------------------------
FUNCTION MLT_t1_t2 (
  crisp  IN NUMBER,
  OBJ    IN FUZZY_LABEL_DEF.OBJ#%TYPE, COL IN FUZZY_LABEL_DEF.OBJ#%TYPE,
  F_TYPE IN FUZZY_COL_LIST.F_TYPE%TYPE,
  F1 IN NUMBER, F2 IN NUMBER, F3 IN NUMBER ,F4 IN NUMBER,
  margen IN FUZZY_APPROX_MUCH.MARGEN%TYPE,
  much   IN FUZZY_APPROX_MUCH.MUCH%TYPE) RETURN NUMBER
IS
  alfa FUZZY_LABEL_DEF.ALFA%TYPE;
  beta FUZZY_LABEL_DEF.ALFA%TYPE;
BEGIN
  IF    F_TYPE=0 THEN RETURN MayorMenor_UNKNOWN(3); -- 3 es el tipo crisp
  ELSIF F_TYPE=1 THEN RETURN MayorMenor_UNDEFINED;
  ELSIF F_TYPE=2 THEN RETURN MayorMenor_NULL(3);
  ELSIF F_TYPE=3 THEN RETURN MLT_t1_t1(crisp,F1,margen,much);
  ELSIF F_TYPE=4 THEN --MLT_crisp_label(crisp,OBJ,COL,F1,margen,much);
    SELECT ALFA,BETA INTO alfa,beta FROM FUZZY_LABEL_DEF
      WHERE OBJ#=OBJ AND COL#=COL AND FUZZY_ID=F1;
    IF    crisp      <=alfa-much THEN RETURN 1;
    ELSIF crisp-margen<beta-much THEN RETURN ROUND( (beta-much-crisp+margen)/(beta-alfa+margen) ,2);
    ELSE  RETURN 0;
    END IF;
  ELSIF F_TYPE=5 THEN RETURN MLT_crisp_finter(crisp,F1,F4,margen,much);
  ELSIF F_TYPE=6 THEN RETURN MLT_crisp_aprox(crisp,F1,F2,F3,F4,margen,much);
  ELSE                RETURN MLT_crisp_trape(crisp,F1,F2,F3,F4,margen,much);
  END IF;
END MLT_t1_t2;

--********************************************************************
-- Indica si una columna difusa tipo 2 es 'MUCHO MENOR' (MLT) que un valor crisp
-- Caso ejemplo: fcol_t2 MLT 8. Tambin: fcol_t2 MLT fcol_t1
--********************************************************************
FUNCTION MLT_crisp (
  crisp  IN NUMBER,
  OBJ    IN FUZZY_LABEL_DEF.OBJ#%TYPE, COL IN FUZZY_LABEL_DEF.OBJ#%TYPE,
  F_TYPE IN FUZZY_COL_LIST.F_TYPE%TYPE,
  F1 IN NUMBER, F2 IN NUMBER, F3 IN NUMBER ,F4 IN NUMBER,
  margen IN FUZZY_APPROX_MUCH.MARGEN%TYPE,
  much   IN FUZZY_APPROX_MUCH.MUCH%TYPE) RETURN NUMBER
IS
  alfa FUZZY_LABEL_DEF.ALFA%TYPE;
  beta FUZZY_LABEL_DEF.ALFA%TYPE;
BEGIN
  IF    F_TYPE=0 THEN RETURN MayorMenor_UNKNOWN(3); -- 3 es el tipo crisp
  ELSIF F_TYPE=1 THEN RETURN MayorMenor_UNDEFINED;
  ELSIF F_TYPE=2 THEN RETURN MayorMenor_NULL(3);
  ELSIF F_TYPE IN (3,5,6) THEN
    -- Comparaciones: F1 MLT crisp, [F1,F4] MLT crisp y #F1 MLT crisp
    RETURN MLT_t1_t1(F1,crisp,margen,much);
  ELSIF F_TYPE=4 THEN -- Comparacin $label MLT crisp
    SELECT ALFA,BETA INTO alfa,beta FROM FUZZY_LABEL_DEF
      WHERE OBJ#=OBJ AND COL#=COL AND FUZZY_ID=F1;
    IF    beta<=crisp-margen-much THEN RETURN 1;
    ELSIF alfa< crisp-much THEN RETURN ROUND( (crisp-much-alfa)/(beta-alfa+margen) ,2);
    ELSE  RETURN 0;
    END IF;
  ELSE  -- Comparacin $[F1,F2+F1,F3+F4,F4] MLT crisp
    IF    F1+F2<=crisp-margen-much THEN RETURN 1;
    ELSIF F1< crisp-much THEN RETURN ROUND( (crisp-much-F1)/(F2+margen) ,2);
    ELSE  RETURN 0;
    END IF;
  END IF;
END MLT_crisp;

--********************************************************************
-- Indica si una columna difusa tipo 2 es 'MUCHO MENOR' (MLT) que un intervalo [X,Y]
-- Caso ejemplo: fcol_t2 MLT [X,Y]
--********************************************************************
FUNCTION MLT_inter (
  X IN NUMBER, Y IN NUMBER,
  OBJ    IN FUZZY_LABEL_DEF.OBJ#%TYPE, COL IN FUZZY_LABEL_DEF.OBJ#%TYPE,
  F_TYPE IN FUZZY_COL_LIST.F_TYPE%TYPE,
  F1 IN NUMBER, F2 IN NUMBER, F3 IN NUMBER, F4 IN NUMBER,
  margen IN FUZZY_APPROX_MUCH.MARGEN%TYPE,
  much   IN FUZZY_APPROX_MUCH.MUCH%TYPE) RETURN NUMBER
IS
  alfa FUZZY_LABEL_DEF.ALFA%TYPE;
  beta FUZZY_LABEL_DEF.ALFA%TYPE;
BEGIN
  IF    F_TYPE=0 THEN RETURN MayorMenor_UNKNOWN(3); -- 3 es el tipo crisp
  ELSIF F_TYPE=1 THEN RETURN MayorMenor_UNDEFINED;
  ELSIF F_TYPE=2 THEN RETURN MayorMenor_NULL(3);
  ELSIF F_TYPE IN (3,5) THEN
     -- Comparaciones: F1 MLT [X,Y] y [F1,F4] MLT [X,Y]
    RETURN MLT_t1_t1(F1,X,margen,much);
  ELSIF F_TYPE=4 THEN -- Comparacin $label MLT [X,Y]
    SELECT ALFA,BETA INTO alfa,beta FROM FUZZY_LABEL_DEF
      WHERE OBJ#=OBJ AND COL#=COL AND FUZZY_ID=F1;
    IF    beta<=X-margen-much THEN RETURN 1;
    ELSIF alfa< X-much THEN RETURN ROUND( (X-much-alfa)/(beta-alfa+margen) ,2);
    ELSE  RETURN 0;
    END IF;
  ELSIF F_TYPE=6 THEN -- Comparacin #F1 MLT [X,Y]
    IF    F1<=X-margen-much THEN RETURN 1;
    ELSIF F2< X-much        THEN RETURN ROUND( (X-much-F2)/(F4+margen) ,2);
    ELSE  RETURN 0;
    END IF;
  ELSE  -- Comparacin $[F1,F2+F1,F3+F4,F4] MLT [X,Y]
    IF    F1+F2<=X-margen-much THEN RETURN 1;
    ELSIF F1< X-much THEN RETURN ROUND( (X-much-F1)/(F2+margen) ,2);
    ELSE  RETURN 0;
    END IF;
  END IF;
END MLT_inter;

--********************************************************************
-- Indica si una columna difusa tipo 2 es 'MUCHO MENOR' (MLT) que un aprox [A1,A2,A3,A4]
-- Caso ejemplo: fcol_t2 MLT #A1
--********************************************************************
FUNCTION MLT_aprox (
  A1 IN NUMBER, A2 IN NUMBER, A3 IN NUMBER, A4 IN NUMBER,
  OBJ    IN FUZZY_LABEL_DEF.OBJ#%TYPE, COL IN FUZZY_LABEL_DEF.OBJ#%TYPE,
  F_TYPE IN FUZZY_COL_LIST.F_TYPE%TYPE,
  F1 IN NUMBER, F2 IN NUMBER, F3 IN NUMBER, F4 IN NUMBER,
  margen IN FUZZY_APPROX_MUCH.MARGEN%TYPE,
  much   IN FUZZY_APPROX_MUCH.MUCH%TYPE) RETURN NUMBER
IS
  alfa FUZZY_LABEL_DEF.ALFA%TYPE;
  beta FUZZY_LABEL_DEF.ALFA%TYPE;
BEGIN
  IF    F_TYPE=0 THEN RETURN MayorMenor_UNKNOWN(3); -- 3 es el tipo crisp
  ELSIF F_TYPE=1 THEN RETURN MayorMenor_UNDEFINED;
  ELSIF F_TYPE=2 THEN RETURN MayorMenor_NULL(3);
  ELSIF F_TYPE IN (3,5) THEN -- Comparaciones: F1 MLT #A1 y [F1,F4] MLT #A1
    RETURN MLT_crisp_aprox(F1,A1,A2,A3,A4,margen,much);
  ELSIF F_TYPE=4 THEN -- Comparacin $label MLT #A1
    SELECT ALFA,BETA INTO alfa,beta FROM FUZZY_LABEL_DEF
      WHERE OBJ#=OBJ AND COL#=COL AND FUZZY_ID=F1;
    IF    beta<=A2-much THEN RETURN 1;
    ELSIF alfa< A1-much THEN RETURN ROUND( (A1-much-alfa)/(beta-alfa+A4) ,2);
    ELSE  RETURN 0;
    END IF;
  ELSIF F_TYPE=6 THEN -- Comparacin #F1 MLT #A1
    IF    F1<=A2-much THEN RETURN 1;
    ELSIF F2< A1-much THEN RETURN ROUND( (A1-much-F2)/(F4+A4) ,2);
    ELSE  RETURN 0;
    END IF;
  ELSE  -- Comparacin $[F1,F2+F1,F3+F4,F4] MLT #A1
    IF    F1+F2<=A2-much THEN RETURN 1;
    ELSIF F1   < A1-much THEN RETURN ROUND( (A1-much-F1)/(F2+A4) ,2);
    ELSE  RETURN 0;
    END IF;
  END IF;
END MLT_aprox;

--********************************************************************
-- Indica si una columna difusa tipo 2 es 'MUCHO MENOR' (MLT) que un trapecio
-- Caso ejemplo: fcol_t2 MLT $[T1,T2,T3,T4]
--********************************************************************
FUNCTION MLT_trape (
  T1 IN NUMBER, T2 IN NUMBER, T3 IN NUMBER, T4 IN NUMBER,
  OBJ    IN FUZZY_LABEL_DEF.OBJ#%TYPE, COL IN FUZZY_LABEL_DEF.OBJ#%TYPE,
  F_TYPE IN FUZZY_COL_LIST.F_TYPE%TYPE,
  F1 IN NUMBER, F2 IN NUMBER, F3 IN NUMBER, F4 IN NUMBER,
  margen IN FUZZY_APPROX_MUCH.MARGEN%TYPE,
  much   IN FUZZY_APPROX_MUCH.MUCH%TYPE) RETURN NUMBER
IS
  alfa FUZZY_LABEL_DEF.ALFA%TYPE;
  beta FUZZY_LABEL_DEF.ALFA%TYPE;
BEGIN
  IF    F_TYPE=0 THEN RETURN MayorMenor_UNKNOWN(3); -- 3 es el tipo crisp
  ELSIF F_TYPE=1 THEN RETURN MayorMenor_UNDEFINED;
  ELSIF F_TYPE=2 THEN RETURN MayorMenor_NULL(3);
  ELSIF F_TYPE IN (3,5) THEN
    -- Comparaciones: F1 MLT $[F1,T2+T1,T3+T4,T4] y [F1,F4] MLT $[F1,T2+T1,T3+T4,T4]
    RETURN MLT_crisp_trape(F1,T1,T2,T3,T4,margen,much);
  ELSIF F_TYPE=4 THEN -- Comparacin $label MLT #A1
    SELECT ALFA,BETA INTO alfa,beta FROM FUZZY_LABEL_DEF
      WHERE OBJ#=OBJ AND COL#=COL AND FUZZY_ID=F1;
    IF    beta<=T1-much    THEN RETURN 1;
    ELSIF alfa< T1+T2-much THEN RETURN ROUND( (T1+T2-much-alfa)/(beta-alfa+T2) ,2);
    ELSE  RETURN 0;
    END IF;
  ELSIF F_TYPE=6 THEN -- Comparacin #F1 MLT #A1
    IF    F1<=T1-much    THEN RETURN 1;
    ELSIF F2< T1+T2-much THEN RETURN ROUND( (T1+T2-much-F2)/(F4+T2) ,2);
    ELSE  RETURN 0;
    END IF;
  ELSE  -- Comparacin $[F1,F2+F1,F3+F4,F4] MLT #A1
    IF    F1+F2<=T1-much    THEN RETURN 1;
    ELSIF F1   < T1+T2-much THEN RETURN ROUND( (T1+T2-much-F1)/(F2+T2) ,2);
    ELSE  RETURN 0;
    END IF;
  END IF;
END MLT_trape;

--********************************************************************
-- Funcin difusa de comparacin: "MUCHO MENOR QUE": MLT (Much Less Than).
-- Indica si una columna difusa tipo 2 es 'MUCHO MENOR QUE' (MLT) otra
-- Caso ejemplo: fcol1_t2 MLT fcol2_t2
--********************************************************************
FUNCTION MLT (
  OBJ1    IN FUZZY_LABEL_DEF.OBJ#%TYPE, COL1 IN FUZZY_LABEL_DEF.OBJ#%TYPE,
  F_TYPE1 IN FUZZY_COL_LIST.F_TYPE%TYPE,
  F11 IN NUMBER, F21 IN NUMBER, F31 IN NUMBER, F41 IN NUMBER,
  OBJ2    IN FUZZY_LABEL_DEF.OBJ#%TYPE, COL2 IN FUZZY_LABEL_DEF.OBJ#%TYPE,
  F_TYPE2 IN FUZZY_COL_LIST.F_TYPE%TYPE,
  F12 IN NUMBER, F22 IN NUMBER, F32 IN NUMBER ,F42 IN NUMBER,
  margen IN FUZZY_APPROX_MUCH.MARGEN%TYPE,
  much   IN FUZZY_APPROX_MUCH.MUCH%TYPE) RETURN NUMBER
IS
  alfa  FUZZY_LABEL_DEF.ALFA%TYPE;  gamma FUZZY_LABEL_DEF.ALFA%TYPE;
  beta  FUZZY_LABEL_DEF.ALFA%TYPE;  delta FUZZY_LABEL_DEF.ALFA%TYPE;
BEGIN
  IF    F_TYPE2=0 THEN RETURN MayorMenor_UNKNOWN(F_TYPE1);
  ELSIF F_TYPE2=1 THEN RETURN MayorMenor_UNDEFINED;
  ELSIF F_TYPE2=2 THEN RETURN MayorMenor_NULL(F_TYPE1);
  ELSIF F_TYPE1=0 THEN RETURN MayorMenor_UNKNOWN(F_TYPE2);
  ELSIF F_TYPE1=1 THEN RETURN MayorMenor_UNDEFINED;
  ELSIF F_TYPE1=2 THEN RETURN MayorMenor_NULL(F_TYPE2);
  ELSIF F_TYPE2=3 THEN -- CRISP
    RETURN MLT_crisp(F12,OBJ1,COL1,F_TYPE1,F11,F21,F31,F41,margen,much);
  ELSIF F_TYPE2=4 THEN -- LABEL: Calculamos los valores del trapecio
    SELECT ALFA,BETA,GAMMA,DELTA INTO alfa,beta,gamma,delta FROM FUZZY_LABEL_DEF
      WHERE OBJ#=OBJ2 AND COL#=COL2 AND FUZZY_ID=F12;
    RETURN MLT_trape(alfa,beta-alfa,gamma-delta,delta,OBJ1,COL1,F_TYPE1,F11,F21,F31,F41,margen,much);
  ELSIF F_TYPE2=5 THEN -- INTERVALO
    RETURN MLT_inter(F12,F42,OBJ1,COL1,F_TYPE1,F11,F21,F31,F41,margen,much);
  ELSIF F_TYPE2=6 THEN -- APROX
    RETURN MLT_aprox(F12,F22,F32,F42,OBJ1,COL1,F_TYPE1,F11,F21,F31,F41,margen,much);
  ELSE -- F_TYPE2=7: TRAPECIO: alfa=F1, delta=F4, beta:=F2+F1; gamma:=F3+F4;
    RETURN MLT_trape(F12,F22,F32,F42,OBJ1,COL1,F_TYPE1,F11,F21,F31,F41,margen,much);
  END IF;
END MLT;


--********************** Comparador difuso NMGT **********************--

----------------------------------------------------------------------
-- Indica la Necesidad de que un valor crisp sea 'MUCHO MAYOR' (NMGT)
-- que un intervalo [X,Y].
-- El valor crisp se difumina a un APROX y el intervalo a un trapecio.
-- Caso ejemplo: fcol_t1 NMGT [3,8]
----------------------------------------------------------------------
FUNCTION NMGT_crisp_finter (
  crisp  IN NUMBER, X IN NUMBER, Y IN NUMBER,
  margen IN FUZZY_APPROX_MUCH.MARGEN%TYPE,
  much   IN FUZZY_APPROX_MUCH.MUCH%TYPE) RETURN NUMBER IS
BEGIN
  -- Comparacin: #crisp NMGT $[X+much-margen,X+much,Y+much,Y+much+margen]
  IF    crisp-margen>=Y+much+margen THEN RETURN 1;
  ELSIF crisp>Y+much THEN
    RETURN ROUND( (crisp-Y-much)/(margen+margen) ,2);
  ELSE  RETURN 0;
  END IF;
END NMGT_crisp_finter;

----------------------------------------------------------------------
-- Indica la Necesidad de que un valor crisp sea 'MUCHO MAYOR' (NMGT)
-- que un valor aprox. Caso ejemplo: fcol_t1 FGT #6
-- Valor aprox:  F1  F4 (donde F4 es el margen: F1-F4=F2 y F1+F4=F3)
----------------------------------------------------------------------
FUNCTION NMGT_crisp_aprox (
  crisp  IN NUMBER,
  F1 IN NUMBER, F2 IN NUMBER, F3 IN NUMBER, F4 IN NUMBER,
  margen IN FUZZY_APPROX_MUCH.MARGEN%TYPE,
  much   IN FUZZY_APPROX_MUCH.MUCH%TYPE) RETURN NUMBER IS
BEGIN
  -- Comparacin: crispmargen FGT (F1+much)F4
  IF    crisp-margen>=F3+much THEN RETURN 1;
  ELSIF crisp>F1+much THEN
    RETURN ROUND( (crisp-F1-much)/(F4+margen) ,2);
  ELSE  RETURN 0;
  END IF;
END NMGT_crisp_aprox;

----------------------------------------------------------------------
-- Indica la Necesidad de que un valor crisp sea 'MUCHO MAYOR' (NMGT)
-- que un trapecio. Caso ejemplo: fcol_t1 NMGT $[1,4,7,9]
----------------------------------------------------------------------
FUNCTION NMGT_crisp_trape (
  crisp  IN NUMBER,
  F1 IN NUMBER, F2 IN NUMBER, F3 IN NUMBER ,F4 IN NUMBER,
  margen IN FUZZY_APPROX_MUCH.MARGEN%TYPE,
  much   IN FUZZY_APPROX_MUCH.MUCH%TYPE) RETURN NUMBER IS
BEGIN
  -- alfa=F1, delta=F4, beta:=F2+F1; gamma:=F3+F4;
  IF    crisp-margen>=F4+much THEN RETURN 1;
  ELSIF crisp>F3+F4+much THEN
    RETURN ROUND( (crisp-F3-F4-much)/(margen-F3) ,2);
  ELSE  RETURN 0;
  END IF;
END NMGT_crisp_trape;

----------------------------------------------------------------------
-- Indica la Necesidad de que un valor crisp sea 'MUCHO MAYOR' (NMGT) que una etiqueta.
-- No se exporta (caso fcol_t1 NMGT $label se traduce a NMGT_crisp_trape)
----------------------------------------------------------------------
FUNCTION NMGT_crisp_label (
  crisp  IN NUMBER,
  OBJ    IN FUZZY_LABEL_DEF.OBJ#%TYPE, COL IN FUZZY_LABEL_DEF.OBJ#%TYPE,
  F_ID   IN FUZZY_LABEL_DEF.FUZZY_ID%TYPE,
  margen IN FUZZY_APPROX_MUCH.MARGEN%TYPE,
  much   IN FUZZY_APPROX_MUCH.MUCH%TYPE) RETURN NUMBER
IS
  gamma  FUZZY_LABEL_DEF.ALFA%TYPE;
  delta  FUZZY_LABEL_DEF.ALFA%TYPE;
BEGIN
  SELECT GAMMA,DELTA INTO gamma,delta FROM FUZZY_LABEL_DEF
    WHERE OBJ#=OBJ AND COL#=COL AND FUZZY_ID=F_ID;
  IF    crisp-margen>=delta+much THEN RETURN 1;
  ELSIF crisp>gamma+much THEN
    RETURN ROUND( (crisp-delta-much)/(delta-gamma+margen) ,2);
  ELSE  RETURN 0;
  END IF;
END NMGT_crisp_label;

----------------------------------------------------------------------
-- Indica la Necesidad de que una columna difusa tipo 1 (crisp1) sea
-- 'MUCHO MAYOR' (NMGT) que un valor crisp, columna tipo 1 o expresin crisp (crisp2).
-- Formatos: fcol_t1 NMGT fcol_t1, fcol_t1 NMGT crisp2  fcol_t1 NMGT expr_crisp
-- Este comparador es difuso 'per se' (no difuminado a partir de uno crisp)
-- por tanto sus comparaciones deben ser ms difuminadas: Por ello, se difuminan
-- los 2 valores crisp a comparar suponiendo que ambos crisp son valores aprox.
----------------------------------------------------------------------
FUNCTION NMGT_t1_t1 (
  crisp1 IN NUMBER, crisp2 IN NUMBER,
  margen IN FUZZY_APPROX_MUCH.MARGEN%TYPE,
  much   IN FUZZY_APPROX_MUCH.MUCH%TYPE) RETURN NUMBER IS
BEGIN
  IF crisp1-margen>=crisp2+margen+much THEN RETURN 1;
  ELSIF crisp1>crisp2+much THEN
       RETURN ROUND( (crisp1-crisp2-much)/(margen+margen) ,2);
  ELSE RETURN 0;
  END IF;
END NMGT_t1_t1;

----------------------------------------------------------------------
-- Indica la Necesidad de que una columna difusa tipo 1 (crisp) sea
-- 'MUCHO MAYOR' (NMGT) que una columna tipo 2: fcol_t1 NMGT fcol_t2
----------------------------------------------------------------------
FUNCTION NMGT_t1_t2 (
  crisp  IN NUMBER,
  OBJ    IN FUZZY_LABEL_DEF.OBJ#%TYPE, COL IN FUZZY_LABEL_DEF.OBJ#%TYPE,
  F_TYPE IN FUZZY_COL_LIST.F_TYPE%TYPE,
  F1 IN NUMBER, F2 IN NUMBER, F3 IN NUMBER ,F4 IN NUMBER,
  margen IN FUZZY_APPROX_MUCH.MARGEN%TYPE,
  much   IN FUZZY_APPROX_MUCH.MUCH%TYPE) RETURN NUMBER IS
BEGIN
  IF    F_TYPE=0 THEN RETURN MayorMenor_UNKNOWN(3); -- 3 es el tipo crisp
  ELSIF F_TYPE=1 THEN RETURN MayorMenor_UNDEFINED;
  ELSIF F_TYPE=2 THEN RETURN MayorMenor_NULL(3);
  ELSIF F_TYPE=3 THEN RETURN NMGT_t1_t1(crisp,F1,margen,much);
  ELSIF F_TYPE=4 THEN RETURN NMGT_crisp_label(crisp,OBJ,COL,F1,margen,much);
  ELSIF F_TYPE=5 THEN RETURN NMGT_crisp_finter(crisp,F1,F4,margen,much);
  ELSIF F_TYPE=6 THEN RETURN NMGT_crisp_aprox(crisp,F1,F2,F3,F4,margen,much);
  ELSE                RETURN NMGT_crisp_trape(crisp,F1,F2,F3,F4,margen,much);
  END IF;
END NMGT_t1_t2;

--********************************************************************
-- Indica la Necesidad de que una columna difusa tipo 2 sea 'MUCHO MAYOR'
-- (NMGT) que un crisp.
-- Caso ejemplo: fcol_t2 NMGT 8. Tambin: fcol_t2 NMGT fcol_t1
--********************************************************************
FUNCTION NMGT_crisp (
  crisp  IN NUMBER,
  OBJ    IN FUZZY_LABEL_DEF.OBJ#%TYPE, COL IN FUZZY_LABEL_DEF.OBJ#%TYPE,
  F_TYPE IN FUZZY_COL_LIST.F_TYPE%TYPE,
  F1 IN NUMBER, F2 IN NUMBER, F3 IN NUMBER ,F4 IN NUMBER,
  margen IN FUZZY_APPROX_MUCH.MARGEN%TYPE,
  much   IN FUZZY_APPROX_MUCH.MUCH%TYPE) RETURN NUMBER
IS
  alfa FUZZY_LABEL_DEF.ALFA%TYPE;
  beta FUZZY_LABEL_DEF.ALFA%TYPE;
BEGIN
  IF    F_TYPE IN (0,1,2) THEN RETURN 0;
  ELSIF F_TYPE=3 THEN RETURN NMGT_t1_t1(F1,crisp,margen,much);
  ELSIF F_TYPE=4 THEN -- $label NMGT #crisp
    SELECT ALFA,BETA INTO alfa,beta FROM FUZZY_LABEL_DEF
      WHERE OBJ#=OBJ AND COL#=COL AND FUZZY_ID=F1;
    IF    alfa>=crisp+margen+much THEN RETURN 1;
    ELSIF beta> crisp+much THEN RETURN ROUND( (beta-crisp-much)/(margen+beta-alfa) ,2);
    ELSE  RETURN 0;
    END IF;
  ELSIF F_TYPE=5 THEN -- Comparacin $[F1-margen,F1,F4,F4+margen] NMGT #crisp
    IF    F1-margen>=crisp+margen+much THEN RETURN 1;
    ELSIF F1>crisp+much THEN RETURN ROUND( (F1-crisp-much)/(margen+margen) ,2);
    ELSE  RETURN 0;
    END IF;
  ELSIF F_TYPE=6 THEN -- Comparacin #F1 NMGT #crisp
    IF    F2>=crisp+margen+much THEN RETURN 1;
    ELSIF F1>crisp+much THEN RETURN ROUND( (F1-crisp-much)/(margen+F4) ,2);
    ELSE  RETURN 0;
    END IF;
  ELSE  -- Comparacin $[F1,F2+F1,F3+F4,F4] NMGT #crisp
    IF    F1>=crisp+margen+much THEN RETURN 1;
    ELSIF F1+F2>crisp+much THEN RETURN ROUND( (F1+F2-crisp-much)/(margen+F2) ,2);
    ELSE  RETURN 0;
    END IF;
  END IF;
END NMGT_crisp;

--********************************************************************
-- Indica la Necesidad de que una columna difusa tipo 2 sea 'MUCHO MAYOR'
-- (NMGT) que un intervalo [X,Y]. Caso ejemplo: fcol_t2 NMGT [X,Y]
--********************************************************************
FUNCTION NMGT_inter (
  X   IN NUMBER, Y IN NUMBER,
  OBJ IN FUZZY_LABEL_DEF.OBJ#%TYPE, COL IN FUZZY_LABEL_DEF.OBJ#%TYPE,
  F_TYPE IN FUZZY_COL_LIST.F_TYPE%TYPE,
  F1 IN NUMBER, F2 IN NUMBER, F3 IN NUMBER, F4 IN NUMBER,
  margen IN FUZZY_APPROX_MUCH.MARGEN%TYPE,
  much   IN FUZZY_APPROX_MUCH.MUCH%TYPE) RETURN NUMBER
IS
  alfa FUZZY_LABEL_DEF.ALFA%TYPE;
  beta FUZZY_LABEL_DEF.ALFA%TYPE;
BEGIN
  IF    F_TYPE IN (0,1,2) THEN RETURN 0;
  ELSIF F_TYPE=3 THEN RETURN NMGT_crisp_finter(F1,X,Y,margen,much);
  ELSIF F_TYPE=4 THEN -- $label NMGT $[X-margen+much,X+much,Y+much,Y+margen+much]
    SELECT ALFA,BETA INTO alfa,beta FROM FUZZY_LABEL_DEF
      WHERE OBJ#=OBJ AND COL#=COL AND FUZZY_ID=F1;
    IF    alfa>=Y+margen+much THEN RETURN 1;
    ELSIF beta> Y+much THEN RETURN ROUND( (beta-Y-much)/(margen+beta-alfa) ,2);
    ELSE  RETURN 0;
    END IF;
  ELSIF F_TYPE=5 THEN -- $[F1-margen,F1,F4,F4+margen] NMGT $[X-margen+much,X+much,Y+much,Y+margen+much]
    IF    F1-margen>=Y+margen+much THEN RETURN 1;
    ELSIF F1>Y+much THEN RETURN ROUND( (F1-Y-much)/(margen+margen) ,2);
    ELSE  RETURN 0;
    END IF;
  ELSIF F_TYPE=6 THEN -- #F1 FGT $[X-margen+much,X+much,Y+much,Y+margen+much]
    IF    F2>=Y+margen+much THEN RETURN 1;
    ELSIF F1> Y+much THEN RETURN ROUND( (F1-Y-much)/(margen+F4) ,2);
    ELSE  RETURN 0;
    END IF;
  ELSE -- F_TYPE=7: $[F1,F2+F1,F3+F4,F4] FGT $[X-margen+much,X+much,Y+much,Y+margen+much]
    IF    F1>=Y+margen+much THEN RETURN 1;
    ELSIF F1+F2>Y+much THEN RETURN ROUND( (F1+F2-Y-much)/(margen+F2) ,2);
    ELSE  RETURN 0;
    END IF;
  END IF;
END NMGT_inter;

--********************************************************************
-- Indica la Necesidad de que una columna difusa tipo 2 sea 'MUCHO MAYOR'
-- (NMGT) que un aprox [A1,A2,A3,A4]. Caso ejemplo: fcol_t2 NMGT #A1
--********************************************************************
FUNCTION NMGT_aprox (
  A1  IN NUMBER, A2 IN NUMBER, A3 IN NUMBER, A4 IN NUMBER,
  OBJ IN FUZZY_LABEL_DEF.OBJ#%TYPE, COL IN FUZZY_LABEL_DEF.OBJ#%TYPE,
  F_TYPE IN FUZZY_COL_LIST.F_TYPE%TYPE,
  F1 IN NUMBER, F2 IN NUMBER, F3 IN NUMBER, F4 IN NUMBER,
  margen IN FUZZY_APPROX_MUCH.MARGEN%TYPE,
  much   IN FUZZY_APPROX_MUCH.MUCH%TYPE) RETURN NUMBER
IS
  alfa FUZZY_LABEL_DEF.ALFA%TYPE;
  beta FUZZY_LABEL_DEF.ALFA%TYPE;
BEGIN
  IF    F_TYPE IN (0,1,2) THEN RETURN 0;
  ELSIF F_TYPE=3 THEN RETURN NMGT_crisp_aprox(F1,A1,A2,A3,A4,margen,much);
  ELSIF F_TYPE=4 THEN -- $label NFGT #A1+much
    SELECT ALFA,BETA INTO alfa,beta FROM FUZZY_LABEL_DEF
      WHERE OBJ#=OBJ AND COL#=COL AND FUZZY_ID=F1;
    IF    alfa>=A3+much THEN RETURN 1;
    ELSIF beta> A1+much THEN RETURN ROUND( (beta-A1-much)/(A4+beta-alfa) ,2);
    ELSE  RETURN 0;
    END IF;
  ELSIF F_TYPE=5 THEN -- $[F1-margen,F1,F4,F4+margen] NFGT #A1+much
   IF    F1-margen>=A3+much THEN RETURN 1;
   ELSIF F1> A1+much THEN RETURN ROUND( (F1-A1-much)/(A4+margen) ,2);
   ELSE  RETURN 0;
   END IF;
  ELSIF F_TYPE=6 THEN -- #F1 NFGT #A1+much
    IF    F2>=A3+much THEN RETURN 1;
    ELSIF F1> A1+much THEN RETURN ROUND( (F1-A1-much)/(A4+F4) ,2);
    ELSE  RETURN 0;
    END IF;
  ELSE -- F_TYPE=7: $[F1,F2+F1,F3+F4,F4] NFGT #A1+much
    IF    F1   >=A3+much THEN RETURN 1;
    ELSIF F2+F1> A1+much THEN RETURN ROUND( (F1+F2-A1-much)/(A4+F2) ,2);
    ELSE  RETURN 0;
    END IF;
  END IF;
END NMGT_aprox;

--********************************************************************
-- Indica la Necesidad de que una columna difusa tipo 2 sea 'MUCHO MAYOR'
-- (NMGT) que un trapecio. Caso ejemplo: fcol_t2 NMGT $[T1,T2,T3,T4]
--********************************************************************
FUNCTION NMGT_trape (
  T1 IN NUMBER, T2 IN NUMBER, T3 IN NUMBER, T4 IN NUMBER,
  OBJ    IN FUZZY_LABEL_DEF.OBJ#%TYPE, COL IN FUZZY_LABEL_DEF.OBJ#%TYPE,
  F_TYPE IN FUZZY_COL_LIST.F_TYPE%TYPE,
  F1 IN NUMBER, F2 IN NUMBER, F3 IN NUMBER, F4 IN NUMBER,
  margen IN FUZZY_APPROX_MUCH.MARGEN%TYPE,
  much   IN FUZZY_APPROX_MUCH.MUCH%TYPE) RETURN NUMBER
IS
  alfa FUZZY_LABEL_DEF.ALFA%TYPE;
  beta FUZZY_LABEL_DEF.ALFA%TYPE;
BEGIN
  -- Trapecio: alfa=T1, delta=T4, beta:=T2+T1; gamma:=T3+T4;
  IF    F_TYPE IN (0,1,2) THEN RETURN 0;
  ELSIF F_TYPE=3 THEN RETURN NMGT_crisp_trape(F1,T1,T2,T3,T4,margen,much);
  ELSIF F_TYPE=4 THEN -- $label NFGT $[T1+much,T2+T1+much,T3+T4+much,T4+much]
    SELECT ALFA,BETA INTO alfa,beta FROM FUZZY_LABEL_DEF
      WHERE OBJ#=OBJ AND COL#=COL AND FUZZY_ID=F1;
    IF    alfa>=T4+much    THEN RETURN 1;
    ELSIF beta> T3+T4+much THEN RETURN ROUND( (beta-T3-T4-much)/(beta-alfa-T3) ,2);
    ELSE  RETURN 0;
    END IF;
  ELSIF F_TYPE=5 THEN -- $[F1-margen,F1,F4,F4+margen] NFGT $[T1,T2+T1,T3+T4,T4]+much
   IF    F1-margen>=T4+much THEN RETURN 1;
   ELSIF F1> T3+T4+much     THEN RETURN ROUND( (F1-T3-T4-much)/(margen-T3) ,2);
   ELSE  RETURN 0;
   END IF;
  ELSIF F_TYPE=6 THEN -- #F1 NFGT $[T1,T2+T1,T3+T4,T4]+much
    IF    F2>=T4+much    THEN RETURN 1;
    ELSIF F1> T3+T4+much THEN RETURN ROUND( (F1-T3-T4-much)/(F4-T3) ,2);
    ELSE  RETURN 0;
    END IF;
  ELSE -- F_TYPE=7: $[F1,F2+F1,F3+F4,F4] NFGT $[T1,T2+T1,T3+T4,T4]+much
    IF    F1   >=T4+much    THEN RETURN 1;
    ELSIF F1+F2> T3+T4+much THEN RETURN ROUND( (F1+F2-T3-T4-much)/(F2-T3) ,2);
    ELSE  RETURN 0;
    END IF;
  END IF;
END NMGT_trape;

--********************************************************************
-- Funcin difusa de comparacin: "NECESIDAD de MUCHO MAYOR QUE": NMGT.
-- Indica la Necesidad de que una columna difusa tipo 2 sea 'MUCHO MAYOR QUE'
-- (NMGT) otra. Caso ejemplo: fcol1_t2 NMGT fcol2_t2
--********************************************************************
FUNCTION NMGT (
  OBJ1    IN FUZZY_LABEL_DEF.OBJ#%TYPE, COL1 IN FUZZY_LABEL_DEF.OBJ#%TYPE,
  F_TYPE1 IN FUZZY_COL_LIST.F_TYPE%TYPE,
  F11 IN NUMBER, F21 IN NUMBER, F31 IN NUMBER, F41 IN NUMBER,
  OBJ2    IN FUZZY_LABEL_DEF.OBJ#%TYPE, COL2 IN FUZZY_LABEL_DEF.OBJ#%TYPE,
  F_TYPE2 IN FUZZY_COL_LIST.F_TYPE%TYPE,
  F12 IN NUMBER, F22 IN NUMBER, F32 IN NUMBER ,F42 IN NUMBER,
  margen IN FUZZY_APPROX_MUCH.MARGEN%TYPE,
  much   IN FUZZY_APPROX_MUCH.MUCH%TYPE) RETURN NUMBER
IS
  alfa  FUZZY_LABEL_DEF.ALFA%TYPE;  gamma FUZZY_LABEL_DEF.ALFA%TYPE;
  beta  FUZZY_LABEL_DEF.ALFA%TYPE;  delta FUZZY_LABEL_DEF.ALFA%TYPE;
BEGIN
  IF    F_TYPE2=0 THEN RETURN MayorMenor_UNKNOWN(F_TYPE1);
  ELSIF F_TYPE2=1 THEN RETURN MayorMenor_UNDEFINED;
  ELSIF F_TYPE2=2 THEN RETURN MayorMenor_NULL(F_TYPE1);
  ELSIF F_TYPE1=0 THEN RETURN MayorMenor_UNKNOWN(F_TYPE2); -- Para ahorrar llamadas
  ELSIF F_TYPE1=1 THEN RETURN MayorMenor_UNDEFINED;
  ELSIF F_TYPE1=2 THEN RETURN MayorMenor_NULL(F_TYPE2);
  ELSIF F_TYPE2=3 THEN -- CRISP
     RETURN NMGT_crisp(F12,OBJ1,COL1,F_TYPE1,F11,F21,F31,F41,margen,much);
  ELSIF F_TYPE2=4 THEN -- LABEL: Calculamos los valores del trapecio
     SELECT ALFA,BETA,GAMMA,DELTA INTO alfa,beta,gamma,delta FROM FUZZY_LABEL_DEF
       WHERE OBJ#=OBJ2 AND COL#=COL2 AND FUZZY_ID=F12;
     RETURN NMGT_trape(alfa,beta-alfa,gamma-delta,delta,OBJ1,COL1,F_TYPE1,F11,F21,F31,F41,margen,much);
  ELSIF F_TYPE2=5 THEN -- INTERVALO
     RETURN NMGT_inter(F12,F42,OBJ1,COL1,F_TYPE1,F11,F21,F31,F41,margen,much);
  ELSIF F_TYPE2=6 THEN -- APROX
     RETURN NMGT_aprox(F12,F22,F32,F42,OBJ1,COL1,F_TYPE1,F11,F21,F31,F41,margen,much);
  ELSE -- F_TYPE2=7: TRAPECIO: alfa=F1, delta=F4, beta:=F2+F1; gamma:=F3+F4;
     RETURN NMGT_trape(F12,F22,F32,F42,OBJ1,COL1,F_TYPE1,F11,F21,F31,F41,margen,much);
  END IF;
END NMGT;

--********************** Comparador difuso NMLT **********************--

----------------------------------------------------------------------
-- Indica la Necesidad de que un valor crisp sea 'MUCHO MENOR' (NMLT)
-- que un intervalo [X,Y]. Caso ejemplo: fcol_t1 NMLT [3,8]
----------------------------------------------------------------------
FUNCTION NMLT_crisp_finter (
  crisp  IN NUMBER,
  X IN NUMBER, Y IN NUMBER,
  margen IN FUZZY_APPROX_MUCH.MARGEN%TYPE,
  much   IN FUZZY_APPROX_MUCH.MUCH%TYPE) RETURN NUMBER IS
BEGIN -- Se puede hacer con una llamada a NMGT (pero es menos eficiente)
  IF    crisp+margen<=X-margen-much THEN RETURN 1;
  ELSIF crisp<X-much THEN RETURN ROUND( (X-much-crisp)/(margen+margen) ,2);
  ELSE  RETURN 0;
  END IF;
END NMLT_crisp_finter;

----------------------------------------------------------------------
-- Mide la Necesidad de que un valor crisp sea 'MUCHO MENOR' (NMLT)
-- que un valor aprox. Caso ejemplo: fcol_t1 NMLT #6
-- Valor aprox:  F1 + - F4 (donde F4 es el margen: F1-F4=F2 y F1+F4=F3)
----------------------------------------------------------------------
FUNCTION NMLT_crisp_aprox (
  crisp  IN NUMBER,
  F1 IN NUMBER, F2 IN NUMBER, F3 IN NUMBER, F4 IN NUMBER,
  margen IN FUZZY_APPROX_MUCH.MARGEN%TYPE,
  much   IN FUZZY_APPROX_MUCH.MUCH%TYPE) RETURN NUMBER IS
BEGIN
  IF    crisp+margen<=F2-much THEN RETURN 1;
  ELSIF crisp< F1-much THEN RETURN ROUND( (F1-much-crisp)/(F4+margen) ,2);
  ELSE  RETURN 0;
  END IF;
END NMLT_crisp_aprox;

----------------------------------------------------------------------
-- Mide la Necesidad de que un valor crisp sea 'MUCHO MENOR' (NMLT)
-- que un trapecio. Caso ejemplo: fcol_t1 NMLT $[1,4,7,9]
----------------------------------------------------------------------
FUNCTION NMLT_crisp_trape (
  crisp  IN NUMBER,
  F1 IN NUMBER, F2 IN NUMBER, F3 IN NUMBER ,F4 IN NUMBER,
  margen IN FUZZY_APPROX_MUCH.MARGEN%TYPE,
  much   IN FUZZY_APPROX_MUCH.MUCH%TYPE) RETURN NUMBER IS
BEGIN -- F_TYPE=7: alfa=F1, delta=F4, beta:=F2+F1; gamma:=F3+F4;
  IF    crisp+margen<=F1-much THEN RETURN 1;
  ELSIF crisp<F1+F2-much THEN RETURN ROUND( (F1+F2-much-crisp)/(F2+margen) ,2);
  ELSE  RETURN 0;
  END IF;
END NMLT_crisp_trape;

----------------------------------------------------------------------
-- Indica si una columna difusa tipo 1 (crisp1) es 'MUCHO MENOR' (NMLT)
-- que un valor crisp u otra columna tipo 1 (crisp2).
-- Formato: fcol_t1 NMLT fcol_t1, o tambin  fcol_t1 NMLT crisp2
----------------------------------------------------------------------
FUNCTION NMLT_t1_t1 (
  crisp1 IN NUMBER, crisp2 IN NUMBER,
  margen IN FUZZY_APPROX_MUCH.MARGEN%TYPE,
  much   IN FUZZY_APPROX_MUCH.MUCH%TYPE) RETURN NUMBER IS
BEGIN
  RETURN NMGT_t1_t1(crisp2,crisp1,margen,much);
END NMLT_t1_t1;

----------------------------------------------------------------------
-- Indica si una columna difusa tipo 1 (crisp) es 'MUCHO MENOR' (NMLT)
-- que una columna tipo 2. Formato: fcol_t1 NMLT fcol_t2
----------------------------------------------------------------------
FUNCTION NMLT_t1_t2 (
  crisp  IN NUMBER,
  OBJ    IN FUZZY_LABEL_DEF.OBJ#%TYPE, COL IN FUZZY_LABEL_DEF.OBJ#%TYPE,
  F_TYPE IN FUZZY_COL_LIST.F_TYPE%TYPE,
  F1 IN NUMBER, F2 IN NUMBER, F3 IN NUMBER ,F4 IN NUMBER,
  margen IN FUZZY_APPROX_MUCH.MARGEN%TYPE,
  much   IN FUZZY_APPROX_MUCH.MUCH%TYPE) RETURN NUMBER
IS
  alfa FUZZY_LABEL_DEF.ALFA%TYPE;
  beta FUZZY_LABEL_DEF.ALFA%TYPE;
BEGIN
  IF    F_TYPE=0 THEN RETURN MayorMenor_UNKNOWN(3); -- 3 es el tipo crisp
  ELSIF F_TYPE=1 THEN RETURN MayorMenor_UNDEFINED;
  ELSIF F_TYPE=2 THEN RETURN MayorMenor_NULL(3);
  ELSIF F_TYPE=3 THEN RETURN NMLT_t1_t1(crisp,F1,margen,much);
  ELSIF F_TYPE=4 THEN
    SELECT ALFA,BETA INTO alfa,beta FROM FUZZY_LABEL_DEF
      WHERE OBJ#=OBJ AND COL#=COL AND FUZZY_ID=F1;
    IF    crisp+margen<=alfa-much THEN RETURN 1;
    ELSIF crisp< beta-much THEN RETURN ROUND(  (beta-crisp-much)/(beta-alfa+margen) ,2);
    ELSE  RETURN 0;
    END IF;
  ELSIF F_TYPE=5 THEN RETURN NMLT_crisp_finter(crisp,F1,F4,margen,much);
  ELSIF F_TYPE=6 THEN RETURN NMLT_crisp_aprox(crisp,F1,F2,F3,F4,margen,much);
  ELSE                RETURN NMLT_crisp_trape(crisp,F1,F2,F3,F4,margen,much);
  END IF;
END NMLT_t1_t2;

--********************************************************************
-- Mide la Necesidad de que una columna difusa tipo 2 sea 'MUCHO MENOR' (NMLT)
-- que un valor crisp. Caso ejemplo: fcol_t2 NMLT 8,  fcol_t2 NMLT fcol_t1
--********************************************************************
FUNCTION NMLT_crisp (
  crisp  IN NUMBER,
  OBJ    IN FUZZY_LABEL_DEF.OBJ#%TYPE, COL IN FUZZY_LABEL_DEF.OBJ#%TYPE,
  F_TYPE IN FUZZY_COL_LIST.F_TYPE%TYPE,
  F1 IN NUMBER, F2 IN NUMBER, F3 IN NUMBER ,F4 IN NUMBER,
  margen IN FUZZY_APPROX_MUCH.MARGEN%TYPE,
  much   IN FUZZY_APPROX_MUCH.MUCH%TYPE) RETURN NUMBER
IS
  alfa FUZZY_LABEL_DEF.ALFA%TYPE;
  beta FUZZY_LABEL_DEF.ALFA%TYPE;
BEGIN
  IF    F_TYPE IN (0,1,2) THEN RETURN 0;
  ELSIF F_TYPE=3 THEN RETURN NMGT_t1_t1(crisp,F1,margen,much);
  ELSIF F_TYPE=4 THEN -- Comparacin $label NMLT crisp
     RETURN NMGT_crisp_label(crisp,OBJ,COL,F1,margen,much);
  ELSIF F_TYPE=5 THEN -- Comparacin [F1,F4] NMLT crisp
     RETURN NMGT_crisp_finter(crisp,F1,F4,margen,much);
  ELSIF F_TYPE=6 THEN -- Comparacin #F1 NMLT crisp
     RETURN NMGT_crisp_aprox(crisp,F1,F2,F3,F4,margen,much);
  ELSE  -- Comparacin $[F1,F2+F1,F3+F4,F4] NMLT crisp
     RETURN NMGT_crisp_trape(crisp,F1,F2,F3,F4,margen,much);
  END IF;
END NMLT_crisp;

--********************************************************************
-- Mide la Necesidad de que una columna difusa tipo 2 sea 'MUCHO MENOR'
-- (NMLT) que un intervalo [X,Y]. Caso ejemplo: fcol_t2 NMLT [X,Y]
--********************************************************************
FUNCTION NMLT_inter (
  X IN NUMBER, Y IN NUMBER,
  OBJ    IN FUZZY_LABEL_DEF.OBJ#%TYPE, COL IN FUZZY_LABEL_DEF.OBJ#%TYPE,
  F_TYPE IN FUZZY_COL_LIST.F_TYPE%TYPE,
  F1 IN NUMBER, F2 IN NUMBER, F3 IN NUMBER, F4 IN NUMBER,
  margen IN FUZZY_APPROX_MUCH.MARGEN%TYPE,
  much   IN FUZZY_APPROX_MUCH.MUCH%TYPE) RETURN NUMBER IS
BEGIN
  RETURN NMGT (NULL,NULL,     5,X,NULL,NULL,Y,
               OBJ ,COL ,F_TYPE,F1,F2,F3,F4,margen,much);
END NMLT_inter;

--********************************************************************
-- Mide la Necesidad de que una columna difusa tipo 2 sea 'MUCHO MENOR'
-- (NMLT) que un aprox [A1,A2,A3,A4]. Caso ejemplo: fcol_t2 NMLT #A1
--********************************************************************
FUNCTION NMLT_aprox (
  A1 IN NUMBER, A2 IN NUMBER, A3 IN NUMBER, A4 IN NUMBER,
  OBJ    IN FUZZY_LABEL_DEF.OBJ#%TYPE, COL IN FUZZY_LABEL_DEF.OBJ#%TYPE,
  F_TYPE IN FUZZY_COL_LIST.F_TYPE%TYPE,
  F1 IN NUMBER, F2 IN NUMBER, F3 IN NUMBER, F4 IN NUMBER,
  margen IN FUZZY_APPROX_MUCH.MARGEN%TYPE,
  much   IN FUZZY_APPROX_MUCH.MUCH%TYPE) RETURN NUMBER IS
BEGIN
  RETURN NMGT (NULL,NULL,     6,A1,A2,A3,A4,
               OBJ ,COL ,F_TYPE,F1,F2,F3,F4,margen,much);
END NMLT_aprox;

--********************************************************************
-- Mide la Necesidad de que una columna difusa tipo 2 sea 'MUCHO MENOR'
-- (NMLT) que un trapecio. Caso ejemplo: fcol_t2 NMLT $[T1,T2,T3,T4]
--********************************************************************
FUNCTION NMLT_trape (
  T1 IN NUMBER, T2 IN NUMBER, T3 IN NUMBER, T4 IN NUMBER,
  OBJ    IN FUZZY_LABEL_DEF.OBJ#%TYPE, COL IN FUZZY_LABEL_DEF.OBJ#%TYPE,
  F_TYPE IN FUZZY_COL_LIST.F_TYPE%TYPE,
  F1 IN NUMBER, F2 IN NUMBER, F3 IN NUMBER, F4 IN NUMBER,
  margen IN FUZZY_APPROX_MUCH.MARGEN%TYPE,
  much   IN FUZZY_APPROX_MUCH.MUCH%TYPE) RETURN NUMBER IS
BEGIN
  RETURN NMGT (NULL,NULL,     7,T1,T2,T3,T4,
               OBJ ,COL ,F_TYPE,F1,F2,F3,F4,margen,much);
END NMLT_trape;

--********************************************************************
-- Funcin difusa de comparacin: "NECESARIAMENTE MUCHO MENOR QUE": NMLT
-- Mide la Necesidad de que una columna difusa tipo 2 sea 'MUCHO MENOR QUE'
-- otra. Caso ejemplo: fcol1_t2 NMLT fcol2_t2
--********************************************************************
FUNCTION NMLT (
  OBJ1    IN FUZZY_LABEL_DEF.OBJ#%TYPE, COL1 IN FUZZY_LABEL_DEF.OBJ#%TYPE,
  F_TYPE1 IN FUZZY_COL_LIST.F_TYPE%TYPE,
  F11 IN NUMBER, F21 IN NUMBER, F31 IN NUMBER, F41 IN NUMBER,
  OBJ2    IN FUZZY_LABEL_DEF.OBJ#%TYPE, COL2 IN FUZZY_LABEL_DEF.OBJ#%TYPE,
  F_TYPE2 IN FUZZY_COL_LIST.F_TYPE%TYPE,
  F12 IN NUMBER, F22 IN NUMBER, F32 IN NUMBER ,F42 IN NUMBER,
  margen IN FUZZY_APPROX_MUCH.MARGEN%TYPE,
  much   IN FUZZY_APPROX_MUCH.MUCH%TYPE) RETURN NUMBER IS
BEGIN
  RETURN NMGT (OBJ2,COL2,F_TYPE2,F12,F22,F32,F42,
               OBJ1,COL1,F_TYPE1,F11,F21,F31,F41,margen,much);
END NMLT;

END FSQL_FUNCTIONS2;
/
